<?php

namespace app\models\web;

use Yii;
use app\models\Paciente;
/**
 * This is the model class for table "cita".
 *
 * @property int $id
 * @property int $id_suscripcion
 * @property int $id_clinica
 * @property string $fecha
 * @property int $id_paciente
 * @property int $id_cita_olivia
 * @property int $estado
 * @property int $tiempo_estimado
 * @property string $comentario
 * @property string $duracion
 * @property string $procedimientosAplicados
 * @property string $fechaDB
 * @property string $fechaFin
 * @property float $costo
 * @property int $tipo_cita
 * @property string $fechaCompleta
 * @property Paciente $paciente
 * @property Suscripcion $suscripcion
 * @property Clinica $clinica
 * @property Feedback[] $feedbacks
 * @property ProcedimientoCitaWeb[] $procedimientosCitaWeb
 * @property Procedimiento $primerProcedimiento
 * @property string $textoEstado
 * @property string $nombrePaciente
 * @property string $nombreClinica
 */
class CitaWeb extends \yii\db\ActiveRecord 
{
    
    const ACTIVA = 0;
    const ACEPTADO = 1;
    const DENEGADO = 3;
    
    const CITA_PRESENCIAL = 0;
    const CITA_EN_LINEA = 1;

    const SCENARIO_SAVE = 'scenariosave';
    const SCENARIO_UPDATE = 'scenarioupdate';
    
    public $hasFormat = false;
    
    /**
     * @var string
     */
    public $captcha;
    
    
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'cita';
    }
    
    /**
     * @return \yii\db\Connection the database connection used by this AR class.
     */
    public static function getDB() {
        return Yii::$app->get('citaWebDB');
    }
    
    public function getCustomScenarios()
    {   
        return [
          self::SCENARIO_SAVE =>  ['id_suscripcion', 'fecha', 'id_paciente', 'estado','tiempo_estimado','id_clinica', 'costo', 'tipo_cita', 'captcha'],
          self::SCENARIO_UPDATE =>  ['id_suscripcion', 'fecha', 'id_paciente', 'estado','tiempo_estimado','id_clinica', 'costo', 'tipo_cita'],
      ];
    }
    
    
    public function scenarios()
    {
        $scenarios = $this->getCustomScenarios();
        return $scenarios;
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        $allscenarios = $this->scenarios();
        return [
            [$allscenarios[self::SCENARIO_SAVE], 'required','message' => 'El campo \'{attribute}\' no puede quedar vacío.', 'on' => self::SCENARIO_SAVE],
            [$allscenarios[self::SCENARIO_UPDATE], 'required','message' => 'El campo \'{attribute}\' no puede quedar vacío.', 'on' => self::SCENARIO_UPDATE],
            [['id_suscripcion', 'id_paciente', 'estado','tiempo_estimado','id_clinica', 'tipo_cita'], 'integer'],
            [['fecha'], 'safe'],
            ['captcha', 'captcha', 'captchaAction'=>'/me/captcha'],
            [['costo'], 'number'],
            [['comentario',], 'string', 'max' => 3000],
            [['id_paciente'], 'exist', 'skipOnError' => true, 'targetClass' => PacienteWeb::className(), 'targetAttribute' => ['id_paciente' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'id_suscripcion' => 'Suscripción',
            'fecha' => 'Fecha',
            'id_paciente' => 'Paciente',
            'estado' => 'Estado',
            'comentario' => '¿Tienes algun comentario adicional?',
            'tiempo_estimado' => 'Tiempo Estimado (En Minutos)',
            'id_clinica' => 'Clínica',
            'duracion' => 'Duración',
            'procedimientosAplicados' => 'Procedimientos',
            'nombrePaciente' => 'Paciente',
            'nombreClinica' => 'Clínica',
            'textoEstado' => 'Estado',
            'costo' => 'Costo'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPaciente()
    {
        return $this->hasOne(PacienteWeb::className(), ['id' => 'id_paciente']);
    }

    /**
     * @return string
     */
    public function getNombrePaciente()
    {
        return $this->paciente->nombreCompleto;
    }

    /**
     * @return string
     */
    public function getNombreClinica()
    {
        return $this->clinica->nombre;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        return $this->hasOne(\app\models\Suscripcion::className(), ['id' => 'id_suscripcion']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinica()
    {
        return $this->hasOne(\app\models\Clinica::className(), ['id' => 'id_clinica']);
    }
    

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getProcedimientosCitaWeb()
    {
        return $this->hasMany(ProcedimientoCitaWeb::className(), ['id_cita' => 'id']);
    }

    /**
     * @return string
     */
    public function getDuracion()
    {
        $horas = intval($this->tiempo_estimado/60);
        return ($horas == 0 ? '' : ($horas . ' hora' . ($horas == 1 ? '' : 's') . ', ')) . $this->tiempo_estimado - ($horas * 60) . ' minutos';
    }
    
    public function getProcedimientosAplicados(){
        $procs =[];
        
        foreach ($this->procedimientoCita as $proCitas){
            $procs[] = $proCitas->procedimiento->nombre;
        }
        return implode(', ', $procs);
    }
    
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat){
                $fecha = $this->fechaDB;
                $this->fecha = $fecha;
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }

    public function getFechaDB(){
        if(!$this->hasFormat){
            return \DateTime::createFromFormat('d/m/Y H:i', $this->fecha)->format('Y-m-d H:i');
        }
        return $this->fecha;
    }

    public function getFechaFin(){
        return \date('Y-m-d H:i',(strtotime($this->fechaDB) + ($this->tiempo_estimado * 60)));
    }
     
    public function afterFind(){
        $fecha = $this->fecha;
        if(!is_null($fecha) && !empty($fecha)){
            $fecha_f = \DateTime::createFromFormat('Y-m-d H:i:s', $fecha)->format('d/m/Y H:i');
            $this->fecha = $fecha_f;
            $this->hasFormat = false;
        }
    }
    
    
    
    public function getTextoEstado(){
        switch ($this->estado){
            case self::ACTIVA:
                return 'Activa';
            case self::ACEPTADO:
                return 'Aceptado';
            case self::DENEGADO:
                return 'Denegado';    
        }
    }
    
    /**
     * @return string
     */
    public function getFechaCompleta(){
        return \date('d/m/Y', strtotime($this->fechaDB)) . ' a las ' . \date('g:i A', strtotime($this->fechaDB));
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPrimerProcedimiento(){
        $prc = $this->procedimientosCitaWeb;
        return $prc[0]->procedimiento;
    }

     
}
