<?php

namespace app\models;

use Yii; 

/**
 * This is the model class for table "suscripcion".
 *
 * @property int $id
 * @property int $id_usuario
 * @property string $fecha_inicio
 * @property int $id_plan
 * @property bool $activa
 * @property string $telefono_notificacion
 * @property string $email
 * @property string $fecha_cancelacion
 * @property bool $stripe_activo
 * @property int $tipo
 *
 * @property Paciente[] $pacientes
 * @property Plan $plan
 * @property Usuario $usuario
 * @property UsuarioSuscripcion[] $usuarioSuscripciones
 * @property Prodecimiento[] $prodecimientos
 * @property HorarioMedico[] $horarios
 * @property Cita[] $citas
 * @property Cita[] $citasNotificadas
 * @property Evento[] $eventos
 * @property Clinica[] $clinicas
 * @property Recordatorio[] $recordatorios
 * @property CuponSuscripcion $cuponSuscripcion
 * @property FirmaMedico $firma
 * @property Pagadito $pagadito
 * @property string $etiquetaTipo
 * @property Pais $pais
 */
class Suscripcion extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    
    const TIPO_INDIVIDUAL = 0;
    const TIPO_JERARQUIZADA = 1;
    const TIPO_ADMIN = 2;

    public static $TIPOS = [0 => 'Individual', 1 => 'Jerarquizada'];
    
    public $hasFormat = false;
    
    public static function tableName()
    {
        return 'suscripcion';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id_usuario', 'fecha_inicio', 'id_plan','stripe_activo','tipo'], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.'],
            [['id_usuario', 'id_plan','tipo','id_pais'], 'integer'],
            [['fecha_inicio', 'fecha_cancelacion'], 'safe'],
            [['activa','stripe_activo'], 'boolean'],
            [['telefono_notificacion'], 'string', 'max' => 50],
            [['email'], 'string', 'max' => 150],
            [['id_pais'], 'exist', 'skipOnError' => true, 'targetClass' => Pais::className(), 'targetAttribute' => ['id_pais' => 'id']],
            [['id_plan'], 'exist', 'skipOnError' => true, 'targetClass' => Plan::className(), 'targetAttribute' => ['id_plan' => 'id']],
            [['id_usuario'], 'exist', 'skipOnError' => true, 'targetClass' => Usuario::className(), 'targetAttribute' => ['id_usuario' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'id_usuario' => 'Usuario',
            'fecha_inicio' => 'Fecha de inicio',
            'id_plan' => 'Plan',
            'activa' => 'Activa',
            'telefono_notificacion' => 'Telefono de notificación',
            'email' => 'Email',
            'fecha_cancelacion' => 'Fecha cancelación',
            'stripe_activo' => 'Stripe Activo',
            'tipo' => 'Tipo',
            'plan.nombre' => 'Plan',
            'etiquetaTipo' => 'Tipo',
            'id_pais' => 'País'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPacientes()
    {
        return $this->hasMany(Paciente::className(), ['id_suscripcion' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getProdecimientos()
    {
        return $this->hasMany(Prodecimiento::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRecordatorios()
    {
        return $this->hasMany(Recordatorio::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getHorarios()
    {
        return $this->hasMany(HorarioMedico::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCitas()
    {
        return $this->hasMany(Cita::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTotalCitasNotificadas($fecha_inicio,$fecha_fin){
        
        $sqlStr = 'CALL spTotalNotificaciones(' . $this->id . ',\'' . $fecha_inicio . '\',\'' . $fecha_fin . '\')';
        $result = \yii::$app->db->createCommand($sqlStr)->queryScalar();
        return intval($result);
    }
    
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTotalMensajesEnviados($fecha_inicio,$fecha_fin){
        $total = BitacoraCitaMensaje::find()
                ->where(['bitacora_cita_mensaje.es_sms' => 1])
                ->andWhere(['bitacora_cita_mensaje.es_correo' => 0])
                ->andWhere(['between','bitacora_cita_mensaje.fecha', $fecha_inicio, $fecha_fin])
                ->join('inner join', 'cita', 'bitacora_cita_mensaje.id_cita=cita.id')
                ->andWhere(['cita.id_suscripcion'=>$this->id])
                ->count();
        
        return $total;
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getEventos()
    {
        return $this->hasMany(Evento::className(), ['id_suscripcion' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPlan()
    {
        return $this->hasOne(Plan::className(), ['id' => 'id_plan']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinicas()
    {
        return $this->hasMany(Clinica::className(), ['id_suscripcion' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUsuario()
    {
        return $this->hasOne(Usuario::className(), ['id' => 'id_usuario']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUsuarioSuscripciones()
    {
        return $this->hasMany(UsuarioSuscripcion::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCuponSuscripcion()
    {
        return $this->hasOne(CuponSuscripcion::className(), ['id_suscripcion' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getFirma()
    {
        return $this->hasOne(FirmaMedico::className(), ['id_suscripcion' => 'id']);
    }
    
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPagadito()
    {
        return $this->hasOne(Pagadito::className(), ['id_suscripcion' => 'id']);
    }
    
     /**
     * Gets query for [[Pais]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getPais()
    {
        return $this->hasOne(Pais::className(), ['id' => 'id_pais']);
    }
    
    public function getEtiquetaTipo(){
        return self::$TIPOS[$this->tipo];
    }
    
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat){
                $fecha_inicio = \DateTime::createFromFormat('d/m/Y', $this->fecha_inicio)->format('Y-m-d');
                $this->fecha_inicio = $fecha_inicio;
                if(!empty($this->fecha_cancelacion)){
                    $fecha_cancelacion = \DateTime::createFromFormat('d/m/Y', $this->fecha_cancelacion)->format('Y-m-d');
                    $this->fecha_cancelacion = $fecha_cancelacion;
                }
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }
    
    public function beforeDelete(){
        if($this->getCitas()->count() > 0){
            return false;
        }
        Recordatorio::deleteAll(['id_suscripcion' => $this->id]);
        ConfiguracionSuscripcion::deleteAll(['id_suscripcion' => $this->id]);
        return parent::beforeDelete();
    }
}
