<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "Nota".
 *
 * @property int $id
 * @property string $asunto
 * @property string $comentarios
 * @property string $fecha_inicio
 * @property string $fecha_fin
 * @property int $id_suscripcion
 * @property bool $es_paciente
 * @property int $id_paciente
 * @property bool $privacidad
 * @property bool $finalizado
 * @property bool $todo_dia
 * @property string $fechaInicioDB
 * @property string $fechaFinDB
 * @property string $asuntoRecordatorio
 * @property string $cometariosRecordatorio
 * @property string $fechaFinRecordatorio
 * @property string $nombrePaciente
 * @property Suscripcion $suscripcion
 * @property Paciente $paciente
 * @property string $tiempo
 * @property int $id_clinica
 * @property Clinica $clinica
 */
class RecordatorioDoctor extends \yii\db\ActiveRecord
{
    public $hasFormat = false;
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'recordatorio_doctor';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['asunto', 'id_suscripcion','privacidad','fecha_inicio'], 'required'],
            [['fecha_inicio','fecha_fin'], 'safe'],
            [['id_suscripcion','privacidad', 'id_paciente','id_clinica'], 'integer'],
            [['finalizado','todo_dia', 'es_paciente'], 'boolean'],
            [['asunto'], 'string', 'max' => 200],
            [['comentarios'], 'string', 'max' => 500],
            [['id_suscripcion'], 'exist', 'skipOnError' => true, 'targetClass' => Suscripcion::className(), 'targetAttribute' => ['id_suscripcion' => 'id']],
            [['fecha_fin'],'compareDates']
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'asunto' => 'Asunto',
            'comentarios' => 'Comentarios',
            'fecha_inicio' => 'Fecha inicio',
            'fecha_fin' => 'Fecha fin',
            'id_suscripcion' => 'Id Suscripcion',
            'finalizado' => 'Finalizado',
            'privacidad' => 'Es privado',
            'todo_dia' => 'Todo el día',
            'es_paciente' => 'Agregar paciente',
            'id_paciente' => 'ID paciente'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        return $this->hasOne(Suscripcion::className(), ['id' => 'id_suscripcion']);
    }
    
     /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinica()
    {
        return $this->hasOne(Clinica::className(), ['id' => 'id_clinica']);
    }
    
    
    public function getPaciente(){
        return \app\models\Paciente::findOne($this->id_paciente);
    }
    
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat){
                $fechai = \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_inicio . ($this->todo_dia ? ' 00:00' : ''))
                        ->format('Y-m-d H:i');
                if($this->todo_dia){
                    $this->fecha_fin = \date('d/m/Y H:i', strtotime(\date('Y-m-d', strtotime($fechai)) . ' 23:59'));
                }
                if(!$this->es_paciente){
                    $this->id_paciente = null;
                }
                if($this->todo_dia){
                   $fechaf = \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_fin)->format('Y-m-d H:i');
                } else {
                   $fecha_f_add = strtotime($fechai.'+30 minutes');
                   $fechaf = date('Y-m-d H:i', $fecha_f_add);
                }
                $this->fecha_inicio = $fechai;
                $this->fecha_fin = $fechaf;
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }
    
    public function afterFind(){
        $fechai = $this->fecha_inicio;
        $fechaf = $this->fecha_fin;
        if(!is_null($fechai) && !empty($fechai) && !is_null($fechaf) && !empty($fechaf)){
            $fecha_i = \DateTime::createFromFormat('Y-m-d H:i:s', $fechai)->format('d/m/Y H:i');
            $this->fecha_inicio = $fecha_i;
            $fecha_f = \DateTime::createFromFormat('Y-m-d H:i:s', $fechaf)->format('d/m/Y H:i');
            $this->fecha_fin = $fecha_f;
            $this->hasFormat = false;
        }
    }
    
    public function compareDates()
    {
        if(!$this->todo_dia){
            $start_date = $this->fecha_inicio;
            if(empty($this->fecha_fin)){
                $this->addError('fecha_fin', 'La fecha fin no puede estar vacía.');
                return;
            }
            $end_date = $this->fecha_fin;
            if(!$this->hasFormat){
                $start_date = \DateTime::createFromFormat('d/m/Y H:i', $start_date)->format('Y-m-d H:i');
                $end_date = \DateTime::createFromFormat('d/m/Y H:i', $end_date)->format('Y-m-d H:i');
            }
            $start_time = strtotime($start_date);
            $end_time = strtotime($end_date);
            if (!$this->hasErrors('fecha_fin') && $end_time < $start_time) {
                $this->addError('fecha_fin', 'La fecha fin no es válida.');
            }
        }
    }
    
    public function getFechaInicioDB(){
        if(!$this->hasFormat){
            $fechaI = $this->fecha_inicio;
            return \DateTime::createFromFormat('d/m/Y H:i', $fechaI)->format('Y-m-d H:i');
        }
        return $this->fecha_inicio;
    }
    
    public function getFechaFinDB(){
        if(!$this->hasFormat){
            return \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_fin)->format('Y-m-d H:i');
        }
        return $this->fecha_fin;
    }
    
    
    public function getAsuntoRecordatorio(){
        $usuario = Yii::$app->user->identity;
        $esDoctor = $usuario->tipo == Usuario::DOCTOR;
        return $this->privacidad == 0 || $esDoctor ? $this->asunto : "Recordatorio privado";
    }
    
    
    public function getComentariosRecordatorio(){
        $usuario = Yii::$app->user->identity;
        $esDoctor = $usuario->tipo == Usuario::DOCTOR;
        return $this->privacidad == 0 || $esDoctor ? $this->comentarios : "Recordatorio privado";
    }
    
    public function getFechaFinRecordatorio(){
        return $this->todo_dia == 1 ? "" : $this->fecha_fin;
    }
    
    public function getNombrePaciente(){
        return $this->id_paciente == '' ? 'No definido' 
                : ( boolval($this->es_paciente) ?  \app\models\Paciente::findOne($this->id_paciente)->nombreCompleto 
                    :  'No definido');
    }
    
    public function getTiempo(){
        $fI = date_create($this->fecha_inicio);
        $fF = date_create($this->fecha_fin);
        $i = date_diff($fI, $fF);
        return $i->format('%H').':'.$i->format('%i');
    }
}
