<?php

namespace app\models;

use Yii;
use yii\db\Expression;
use app\customs\Utils;
use borales\extensions\phoneInput\PhoneInputValidator;
use app\customs\SesionUtils;
/**
 * This is the model class for table "paciente".
 *
 * @property int $id
 * @property int $id_suscripcion
 * @property string $codigo
 * @property string $nombre
 * @property string $apellido
 * @property string $fecha_nacimiento
 * @property string $email
 * @property string $telefono_fijo
 * @property string $celular
 * @property bool $usa_whatsapp
 * @property string $sexo
 * @property string $nombre_encargado
 * @property int $id_tipo_encargado
 * @property int $id_pais
 * @property string $direccion
 * @property string $aseguradora
 * @property string $no_aseguradora
 * @property string $notas
 * @property string $nombreCompleto
 * @property string $nombreCorto
 * @property string $primerApellido
 * @property string $primerNombre
 * @property string $nombreCompletoEncargado
 * @property string $nombreCortoEncargado
 * @property string $referido_por
 * @property string $enfermedades_actuales 
 * @property string $alergias
 * @property string $medicamentos
 * @property string $edad
 * @property string $fecha_creacion
 * @property string $nombre_encargado_sec
 * @property string $apellido_encargado_sec
 * @property string $celular_encargado_sec
 * @property string $email_encargado_sec
 * @property int $id_clinica
 * @property int $estado
 * @property string $procedencia
 * @property int $id_grupo
 * 
 * @property Pais $pais
 * @property Cita $citaProxima 
 * @property Cita[] $citas 
 * @property Suscripcion $suscripcion
 * @property TipoEncargado $tipoEncargado
 * @property Adjunto[] $adjuntosRecientes
 * @property Cita $ultimaCita
 * @property Cita $primeraCita
 * @property string $citaProxOrUlt
 * @property Clinica $clinica
 * @property GrupoTrabajo $grupo
 */
class Paciente extends \yii\db\ActiveRecord
{
    
    
    public $hasFormat = false;
    
    const SCENARIO_OLIVIA_REGISTER = 'scenario_olivia_register';
    const SCENARIO_WEB_TO_OLIVIA = 'scenario_web_to_olivia';
    
    const ESTADO_EN_ESPERA = 0;
    const ESTADO_ACTIVO = 1;
    const ESTADO_INCOMPLETO = 2;
    const ESTADO_RETIRADO = 3;
    const ESTADO_INACTIVO = 4;
    const ESTADO_ARCHIVADO = 5;
    
    public static $estados = [0 => 'En espera', 1 => 'Activo', 2 => 'Incompleto', 3 => 'Retirado', 4  => 'Inactivo', 5 => 'Archivado'];
    
    public static $estadosDropdown = [
        self::ESTADO_EN_ESPERA => '<span class="kt-badge kt-badge--en-espera kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_EN_ESPERA.'"></span> En espera',
        self::ESTADO_ACTIVO => '<span class="kt-badge kt-badge--activo kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_ACTIVO.'"></span> Activo',
        self::ESTADO_INCOMPLETO => '<span class="kt-badge kt-badge--incompleto kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_INCOMPLETO.'"></span> Incompleto',
        self::ESTADO_RETIRADO => '<span class="kt-badge kt-badge--retirado kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_RETIRADO.'"></span> Retirado',
        self::ESTADO_INACTIVO => '<span class="kt-badge kt-badge--inactivo kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_INACTIVO.'"></span> Inactivo',
        self::ESTADO_ARCHIVADO => '<span class="kt-badge kt-badge--concluido kt-badge--dot kt-badge--lg mr-2" data-estado="'.self::ESTADO_ARCHIVADO.'"></span> Archivado'
    ];


    public static function itemsDropdown($estado, $id){
        $items = '';
        foreach (self::$estadosDropdown as $key => $value ){
            $class = $estado == $key ? 'hide' : ''; 
            $items .= '<a id="estado-'.$key.'" '
                    . 'class="dropdown-item item-estado-paciente '.$class.'" '
                    . 'href="javascript:void(0);" '
                    . 'data-id="'.$id.'" '
                    . 'data-estado="'.$key.'">'.$value.'</a>';
            
        }
        
        return $items;
    }
    
    public function getDropdownEstados($tipoUsuarioJerarquia, $id='estados-paciente'){
        $disabled = $tipoUsuarioJerarquia == UsuarioSuscripcion::TIPO_DEPENDIENTE ? 'disabled="disabled"' : '';
        return '<div class="dropdown">'
                    .'<button  ' . $disabled . ' class="btn btn-secondary dropdown-toggle dropdown-btn" type="button" id="'.$id.'" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">'
                        .self::$estadosDropdown[$this->estado]
                    .'</button>'
                    .'<div class="dropdown-menu" aria-labelledby="dropdownMenuButton">'
                        .self::itemsDropdown($this->estado, $this->id)
                    .'</div>'
                .'</div>';
    }
    
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'paciente';
    }
    
    private function getRequiredParams()
    {
        $valores = explode('|', SesionUtils::configuracionSesion('campos_requeridos_paciente')->valor);
        $reqs = ['id_suscripcion', 'nombre', 'apellido'];
        if( in_array('correo',$valores ) ){
            $reqs[] = 'email';
        }
        if( in_array('celular',$valores) ){
            $reqs[] = 'celular';
        }
        return [
          self::SCENARIO_OLIVIA_REGISTER =>  $reqs,
          self::SCENARIO_WEB_TO_OLIVIA =>  ['id_suscripcion', 'nombre', 'apellido'],
      ];
    }
    
    public function scenarios()
    {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_OLIVIA_REGISTER] = [
            'id_suscripcion',
            'nombre', 
            'apellido',
            'celular', 
            'email', 
            'codigo', 
            'telefono_fijo', 
            'sexo',
            'fecha_nacimiento', 
            'usa_whatsapp',
            'id_tipo_encargado', 
            'id_clinica', 
            'id_pais', 
            'fecha_creacion', 
            'no_aseguradora', 
            'aseguradora',
            'nombre_encargado', 
            'nombre_encargado_sec', 
            'apellido_encargado_sec', 
            'email_encargado_sec',
            'celular_encargado_sec', 
            'direccion', 
            'referido_por', 
            'enfermedades_actuales',
            'alergias', 
            'notas',
            'estado',
            'id_grupo',
            'nivel_socioeconomico',
            'procedencia'
        ];
        $scenarios[self::SCENARIO_WEB_TO_OLIVIA] = ['id_suscripcion', 'nombre', 'apellido'];
        return $scenarios;
    }

    

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        $reqs = $this->getRequiredParams();
        return [
            [$reqs[self::SCENARIO_OLIVIA_REGISTER], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.', 'on' => self::SCENARIO_OLIVIA_REGISTER],
            [$reqs[self::SCENARIO_WEB_TO_OLIVIA], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.', 'on' => self::SCENARIO_WEB_TO_OLIVIA],
            [['id_suscripcion', 'id_tipo_encargado','id_pais','id_clinica'], 'integer'],
            [['fecha_nacimiento', 'fecha_creacion'], 'safe'],
            [['usa_whatsapp'], 'boolean'],
            [['codigo', 'telefono_fijo', 'celular','no_aseguradora'], 'string', 'max' => 50],
            [['nombre', 'apellido'], 'string', 'max' => 100],
            [['email', 'nombre_encargado','aseguradora', 'nombre_encargado_sec', 'apellido_encargado_sec', 'email_encargado_sec'], 'string', 'max' => 150],
            [['sexo'], 'string', 'max' => 1],
            [['email'],'email'],
            [['celular', 'celular_encargado_sec', 'email_encargado_sec'], 'string'],
            [['telefono_fijo'], PhoneInputValidator::className()],            
            [['celular', 'celular_encargado_sec'], PhoneInputValidator::className()],
            [['direccion'], 'string', 'max' => 500],
            [['referido_por'], 'string', 'max' => 150],            
            [['enfermedades_actuales'], 'string', 'max' => 1000],            
            [['alergias'], 'string', 'max' => 1000],            
            [['medicamentos'], 'string', 'max' => 1000],
            [['notas'], 'string', 'max' => 2000],
            [['id_suscripcion'], 'exist', 'skipOnError' => true, 'targetClass' => Suscripcion::className(), 'targetAttribute' => ['id_suscripcion' => 'id']],
            [['id_tipo_encargado'], 'exist', 'skipOnError' => true, 'targetClass' => TipoEncargado::className(), 'targetAttribute' => ['id_tipo_encargado' => 'id']],
            [['id_pais'], 'exist', 'skipOnError' => true, 'targetClass' => Pais::className(), 'targetAttribute' => ['id_pais' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'id_suscripcion' => 'Suscripción',
            'codigo' => 'Código',
            'nombre' => 'Nombre',
            'apellido' => 'Apellido',
            'fecha_nacimiento' => 'Fecha nacimiento',
            'email' => 'Email',
            'telefono_fijo' => 'Télefono Fijo',
            'celular' => 'Télefono Celular',
            'usa_whatsapp' => 'Usa Whatsapp',
            'sexo' => 'Sexo',
            'nombre_encargado' => 'Nombre encargado',
            'id_tipo_encargado' => 'Tipo de encargado',
            'direccion' => 'Dirección',
            'aseguradora' => 'Aseguradora',
            'no_aseguradora' => 'Número de aseguradora',
            'notas' => 'Notas',
            'nombreCompleto' => 'Nombre completo',
            'referido_por' =>'Referido por',            
            'enfermedades_actuales' =>'Enfermedades actuales',
            'alergias' =>'Alergias',
            'medicamentos' =>'Medicamentos',
            'pais' => 'pais',
            'nombre_encargado_sec' => 'Nombre',
            'apellido_encargado_sec' => 'Apellido',
            'celular_encargado_sec' => 'Télefono Celular',
            'email_encargado_sec' => 'Email',
            'id_clinica' => 'Clínica',
            'id_grupo' => 'Grupo',
            'estado' => 'Estado',
            'nivel_socioeconomico' => 'Nivel Socioeconómico',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        return $this->hasOne(Suscripcion::className(), ['id' => 'id_suscripcion']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPais()
    {
        return $this->hasOne(Pais::className(), ['id' => 'id_pais']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTipoEncargado()
    {
        return $this->hasOne(TipoEncargado::className(), ['id' => 'id_tipo_encargado']);
    }
    
    /**
     * @return string
     */
    public function getNombreCompleto()
    {
        return $this->nombre . ' ' . $this->apellido;
    }
    
    /**
     * @return string
     */
    public function getNombreCompletoEncargado()
    {
        return $this->nombre_encargado_sec . ' ' . $this->apellido_encargado_sec;
    }
    
    /**
     * @return string
     */
    public function getNombreCorto()
    {
        return Utils::shortName($this->nombre,$this->apellido);
    }
    
    /**
     * @return string
     */
    public function getNombreCortoEncargado()
    {
        return Utils::shortName($this->nombre_encargado_sec,$this->apellido_encargado_sec);
    }

    /**
     * @return string
     */
    public function getPrimerApellido()
    {
        return Utils::firstPartName($this->apellido);
    }
    
    /**
     * @return string
     */
    public function getPrimerNombre()
    {
        return Utils::firstPartName($this->nombre);
    }
    
     public function getCitas()
    {
        return $this->hasMany(Cita::className(), ['id_paciente' => 'id'])->orderBy(['fecha'=>SORT_DESC]);
    }
    
    public static function citasCount($id){
        return Cita::find()->where(['id_paciente'=>$id])->count();
    }
    
    public static function listaEsperaCount($id){
        return ListaEspera::find()->where(['id_paciente'=>$id])->count();
    }
    
    public static function historialCount($id){
        return HistorialMedicoPaciente::find()->where(['id_paciente'=>$id])->count();
    }
    
    public static function recetasCount($id){
        return Receta::find()->where(['id_paciente'=>$id])->count();
    }
    
    public static function countDatosPaciente($id){
        $total = 0;
        $idSuscripcion = Yii::$app->user->identity->suscripcion->id;
        try{
            $res = Yii::$app->db->createCommand("CALL spContDatosPaciente(:idPaciente, :idSuscripcion)")
                    ->bindValue('idPaciente', $id)
                    ->bindValue('idSuscripcion', $idSuscripcion)
                    ->queryOne();
            $total = (int) array_sum( array_values( $res ) );
        } catch(\Exception $e){
            $e->getMessage();
            $citas = self::citasCount($id);
            $recetas = self::recetasCount($id);
            $historial = self::historialCount($id);
            $listaEspera = self::listaEsperaCount($id);
            $total = (int) ($citas + $recetas + $historial + $listaEspera); 
        }
        return $total == 0 ? true : false;
    }
    
    public function permitirEliminar(){
        return self::countDatosPaciente($this->id);
    }
    

    public function getHistoricoCitas()
    {
        return $this->hasMany(Cita::className(), ['id_paciente' => 'id'])
                //->andFilterWhere(['<',new Expression('DATE(fecha)'),new Expression('DATE(\'' . \date('Y-m-d') . '\')')])
                ->orderBy(['fecha'=>SORT_DESC]);
    }
    
    public function getCitaProxOrUlt(){
        $citaProx = $this->getCitaProxima();
        $citaAnt = $this->getUltimaCita();
        if($citaProx == null){
            return $citaAnt != null ? 'Ult-'.$citaAnt['fecha'] : 'Sin citas anteriores';
        }
        return $citaProx != null ?'Prox-'.$citaProx['fecha'] : 'Sin citas proximas';    
    }

    public function getCitaProxima()
    {
        //new Expression('date(\'' . \date("Y-m-d H:i:s") . '\')')     
        return Cita::find()->where(['id_paciente' => $this->id])
                ->andWhere(['>=','fecha',date("Y-m-d H:i:s")])
                ->andWhere(['<', 'estado', Cita::CANCELADA])
                ->orderBy(['fecha'=>SORT_ASC])
        ->limit(1)->one();
    }
    
    
    public function getUltimaCita()
    {
        return Cita::find()->where(['id_paciente' => $this->id])
                ->andWhere(['<=', 'fecha', date('Y-m-d H:i:s')])
                ->andWhere(['<>', 'estado', Cita::CANCELADA])
                ->andWhere(['<>', 'estado', Cita::INASISTENCIA])
                ->orderBy(['fecha'=>SORT_DESC])
                ->limit(1)->one();
    }
    
    public function getPrimeraCita(){
        return Cita::find()->where(['id_paciente' => $this->id])
                ->andWhere(['<>', 'estado', Cita::CANCELADA])
                ->andWhere(['<>', 'estado', Cita::INASISTENCIA])
                ->orderBy(['fecha'=>SORT_ASC])
                ->limit(1)->one();
    }
    
    
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinica()
    {
        return $this->hasOne(Clinica::className(), ['id' => 'id_clinica']);
    }
    
    public function afterFind(){
        $fecha = $this->fecha_nacimiento;
        if(!is_null($fecha) && !empty($fecha)){
            $fecha_f = \DateTime::createFromFormat('Y-m-d', $fecha)->format('d/m/Y');
            $this->fecha_nacimiento = $fecha_f;
            $this->hasFormat = false;
        }
    }
    
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat && !is_null($this->fecha_nacimiento) && !empty($this->fecha_nacimiento)){
                $fecha_nacimiento = \DateTime::createFromFormat('d/m/Y', $this->fecha_nacimiento)->format('Y-m-d');
                $this->fecha_nacimiento = $fecha_nacimiento;
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBitacoraMensajePacientes()
    {
        return $this->hasMany(BitacoraMensajePaciente::className(), ['id_paciente' => 'id']);
    }
    
    public function getEdad(){
        $now = date('Y-m-d');
        if($this->fecha_nacimiento != null){
            $F_nac = \DateTime::createFromFormat('d/m/Y', $this->fecha_nacimiento)->format('Y-m-d');
            $fecha_actual = date_create($now);
            $fecha_nac    = date_create($F_nac);
            $diff = date_diff($fecha_nac, $fecha_actual);
            $anios = (int) $diff->format('%y');
            $meses = (int) $diff->format('%m');
            $dias  = (int) $diff->format('%d');
            $labelAnio = $anios > 1 ? 'años' : 'año';
            $labelMes  = $meses > 1 ? 'meses' : 'mes';
            $labelDia  = $dias  > 1 ? 'días' : 'día';
            if($meses >= 1 && $meses <= 5 && $anios < 1){
                return $meses .' '. $labelMes .' '.( $dias == 0 ? '' : $dias).' '.($dias == 0 ? '' : $labelDia);
            } else if($meses >= 6 && $meses < 12 && $anios < 1){
                return $meses .' '. $labelMes;
            } else if($meses < 1 && $anios < 1){
                return $dias .' '. $labelDia;
            } else if($anios <= 5){
                return $anios .' '.$labelAnio.' '.($meses == 0 ? '' : $meses).' '.($meses == 0 ? '' :$labelMes);
            }else {
                return $anios .' '. $labelAnio;
            }
        } else {
            return 'No definido';
        }

    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getGrupo()
    {
        return $this->hasOne(GrupoTrabajo::className(), ['id' => 'id_grupo']);
    }
    
    public function beforeDelete()
    {
        if (!parent::beforeDelete()) {
            return false;
        }
        if($this->getCitas()->count() > 0){
            return false;
        }
        BitacoraMensajePaciente::deleteAll(['id_paciente' => $this->id]);
        return true;
    }
    

    /**
     * Migra el modelo seleccionado al nuevo id
     * @param int $id
     * @return boolean
     */
    public function migrar($id){
        
        try {
            Cita::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            ListaEspera::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            BitacoraPaciente::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            Receta::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            BitacoraMensajePaciente::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            Seguimiento::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            DocumentoPaciente::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
            HistorialMedicoPaciente::updateAll(['id_paciente' => $id],['id_paciente' => $this->id]);
        } catch (Exception $e) {
            return false; // 'Excepción capturada: ',  $e->getMessage(), "\n";
        }
        
        return true;
    }


    public function behaviors() {
        return [
                'ActiveRecordLogableBehavior' => 'app\behaviors\PacienteLogableBehavior',
            ];
    }
    
}
