<?php

namespace app\models;

use yii\base\Model;
use app\models\Cita;
use yii\db\Expression;
use app\models\CitaEvento;
use app\models\Suscripcion;
use app\customs\SesionUtils;

/**
 * Esta es un modelo para el resultado del procedimiento "spFechaPorHorario".
 *
 * @property int $id_clinica
 * @property int $id_suscripcion
 * @property int $dia_semana
 * @property string $fecha
 * @property bool $disponible
 */

class DiaDisponible extends Model
{
    public $dia_semana;
    public $fecha;
    public $disponible;
    public $id_clinica;
    public $id_suscripcion;
    
    public function rules()
    {
        return [
            [['fecha', 'dia_semana'], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.'],
            [['dia_semana'], 'integer'],
            [['fecha'], 'safe']
        ];
    }
    
    public function obtenerCitas(){
        $citas = Cita::find()->where();
    }
    
    /**
     * 
     * @param string $fecha Fecha base para iniciar busqueda.
     * @param int $id_suscripcion Id de la suscripcion actual
     * @param int $id_clinica Clinica seleccionada
     * @param int $estimado tiempo estimado
     * @param int $id_cita id de la cita en edición
     * @param Suscripcion $suscripcion Suscripcion actual
     * @return DiaDisponible[]
     */
    public static function obtenerDiasNoDisponibles($fecha,$id_suscripcion,$id_clinica,$estimado,$id_cita,$suscripcion = null){
        $mes_fecha = intval(\date('m', strtotime($fecha)));
        $mes_actual = intval(\date('m'));
        $fecha_inicio = '';
        if($mes_fecha == $mes_actual){
            $fecha_inicio = \date('Y-m-d');
        }else{
            $date = new \DateTime(\date('Y-m-01',strtotime($fecha)));
            $date->sub(new \DateInterval('P7D'));
            $fecha_inicio = $date->format('Y-m-d');
        }
        $fecha_fin = \date("Y-m-t", strtotime($fecha));
        $date = new \DateTime($fecha_fin);
        $date->add(new \DateInterval('P15D')); // P1D means a period of 1 day
        $fecha_fin = $date->format('Y-m-d');
        
        $sqlStr = "CALL spFechasParaCalendario('" . $fecha_inicio . "','" . $fecha_fin . "'," . $id_suscripcion . "," . $id_clinica . ")";
        $result = \yii::$app->db->createCommand($sqlStr)->queryAll();
        $periodo = new \DatePeriod(new \DateTime($fecha_inicio),new \DateInterval('P1D'),new \DateTime($fecha_fin));
        $fechasPeriodo = [];
        foreach ($periodo as $key => $value) {
            $fechasPeriodo[] = $value->format('Y-m-d');
        }
        return self::parseToDiaDisponible($id_suscripcion,$id_clinica,$estimado,$fechasPeriodo,$id_cita,$suscripcion,$result);
    }
   
    /**
     * 
     * @param date $fecha
     * @param int $estimado
     * @param int $id_suscripcion
     * @param int $id_clinica
     * @return bool
     */
    private static function obtenerDisponibilidadDiaOld($fecha,$estimado,$id_suscripcion,$id_clinica){
        $sqlStr = 'CALL spFechaDisponible(' . $id_suscripcion . ',' . $id_clinica . ',' . $estimado . ',\'' . $fecha . '\')';
        $result = \yii::$app->db->createCommand($sqlStr)->queryScalar();
        return boolval($result);
    }
    
   
    /**
     * 
     * @param string $fecha
     * @param int $estimado
     * @param int $id_suscripcion
     * @param int $id_clinica
     * @param int $id_cita
     * @param Suscripcion $suscripcion
     * @return boolean
     */
    private static function obtenerDisponibilidadDia($fecha,$estimado,$id_suscripcion,$id_clinica,$id_cita,$suscripcion = null){
        $time = time();
        $qHorario = HorarioMedico::find()
                    ->where(['id_suscripcion' => $id_suscripcion])
                    ->andFilterWhere(['id_clinica' => $id_clinica])
                    ->andFilterWhere(['dia' => new Expression('DAYOFWEEK(\'' . $fecha . '\')')]);
        $horarios = $qHorario->all();
        if(count($horarios) == 0){ return FALSE; }
        
        $hora_inicio = $horarios[0]->hora_inicio;
        $hora_fin = $horarios[count($horarios) - 1]->hora_fin;
        
        $shortToday = strtotime(\date('H:i', $time));
        $isToday = ((\date('Y-m-d', strtotime($fecha)) == \date('Y-m-d',$time)));
        if($isToday && $shortToday > strtotime($hora_fin)){return false;}
        
        $eventoLargo = self::eventoLargo($id_suscripcion, $fecha, $hora_inicio, $hora_fin,$suscripcion);
        if ($eventoLargo){ return false; }
        $eventoDia = self::eventoDia($id_suscripcion, $id_clinica, $fecha,$suscripcion);
        if ($eventoDia){ return false; }
        
        foreach($horarios as $horario){
            $horaInicio = $isToday && strtotime($horario->hora_inicio) < $shortToday 
                    ? \date('H:i', $time) 
                    : $horario->hora_inicio;
            $horaFin = $horario->hora_fin;
            $citaEventos = CitaEvento::obtenerCitaEventosPorHorario($id_suscripcion, $id_clinica, $fecha, $horaInicio, $horaFin,$id_cita);
            if(count($citaEventos) == 0){
                return self::timeMinutesDifference($horaInicio,$horaFin) >= $estimado;
            }
            foreach ($citaEventos as $citaEvento) {
                $horaFin = $citaEvento->fecha;
                $minutes = self::timeMinutesDifference($horaInicio,$horaFin);
                if ($minutes >= $estimado){
                    return TRUE;
                }
                $horaInicio = $citaEvento->fecha_fin;
            }
            $horaFin = $horario->hora_fin;
        }
        return self::timeMinutesDifference($horaInicio,$horaFin) >= $estimado;
    }
    
    /**
     * 
     * @param int $id_suscripcion
     * @param string $fecha
     * @param Suscripcion $suscripcion
     * @return bool
     */
    private static function eventoDia($id_suscripcion,$fecha,$suscripcion = null){
        $qEventos = Evento::find()->where(['id_suscripcion' => $id_suscripcion])
                    ->andFilterWhere([
                                    'AND',
                                    ['=',new Expression('DATE(fecha_inicio)'),new Expression("DATE('" . $fecha . "')")],
                                    ['todo_dia' => 1]
                                ]);
        if($suscripcion != null && $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qEventos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        return $qEventos->count() > 0;
    }
    
    /**
     * 
     * @param int $id_suscripcion
     * @param string $fecha
     * @param string $horaInicio
     * @param string $horaFin
     * @param Suscripcion $suscripcion
     * @return bool
     */
    private static function eventoLargo($id_suscripcion,$fecha,$horaInicio,$horaFin,$suscripcion = null){
        $qEventos = Evento::find()->where(['id_suscripcion' => $id_suscripcion])
                    ->andFilterWhere([
                                    'AND',
                                    ['<','fecha_inicio',new Expression("CONCAT('" . $fecha . "',' " . $horaInicio . "')")],
                                    ['>','fecha_fin',new Expression("CONCAT('" . $fecha . "',' " . $horaFin . "')")],
                                ]);
        if($suscripcion != null && $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qEventos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        return $qEventos->count() > 0;
    }
    
    private static function timeMinutesDifference($horaInicio,$horaFin){
        $to_time = strtotime($horaInicio);
        $from_time = strtotime($horaFin);
        $minutes = round(abs($to_time - $from_time) / 60,2);
        return $minutes;
    }
    
    /**
     * 
     * @param int $id_suscripcion
     * @param int $id_clinica
     * @param int $estimado
     * @param array $periodo
     * @param int $id_cita
     * @param Suscripcion $suscripcion
     * @param array $spResult Resultado del SP "spFechaPorHorario"
     * @return DiaDisponible[]
     */
    private static function parseToDiaDisponible($id_suscripcion, $id_clinica,
            $estimado, $periodo,$id_cita,$suscripcion = null, $spResult = []){
        $diaArray = [];
        foreach ($spResult as $r){
            $disponible = self::obtenerDisponibilidadDia($r['fecha'], $estimado, $id_suscripcion, $id_clinica,$id_cita,$suscripcion);
            if($disponible){
                /*
                $horarioHabil = true;
                $time = time();
                if((\date('Y-m-d', strtotime($r['fecha'])) == \date('Y-m-d',$time))){
                    $qHorario = HorarioMedico::find()
                            ->where(['id_suscripcion' => $id_suscripcion])
                            ->andFilterWhere(['id_clinica' => $id_clinica])
                            ->andFilterWhere(['dia' => new Expression('DAYOFWEEK(\'' . \date('Y-m-d',$time) . '\')')])
                            ->andFilterWhere(['BETWEEN',new Expression("TIME('" . \date('H:i',$time) . "')"),new Expression('hora_inicio'),new Expression('hora_fin')]);
                    $horario = $qHorario->count();
                    //echo $qHorario->createCommand()->rawSql;
                    $horarioHabil = $horario > 0;
                }
                if($horarioHabil){
                    $diaArray[] = $r['fecha'];
                }
                 * 
                 */
                $diaArray[] = $r['fecha'];
            }
        }
        return array_diff($periodo,$diaArray) ;
    }
    
    private static function diferenciaHoras($hora_inicio,$hora_fin){
        $minutes = (strtotime($hora_fin) - strtotime($hora_inicio)) / 60;
        return $minutes;
    }
    
    /**
     * 
     * @param app\models\CitaEvento[] $citas
     * @return app\models\CitaEvento
     */
    private static function primerCita($citas){
        if(count($citas)> 0){
            return $citas[0];
        }
        return null;
    }
    
    /**
     * 
     * @param app\models\CitaEvento[] $citas
     * @return app\models\CitaEvento
     */
    private static function ultimaCita($citas){
        if(count($citas)> 0){
            return $citas[count($citas) - 1];
        }
        return null;
    }
    
    /**
     * 
     * @param app\models\CitaEvento[] $citas
     * @param int $estimado
     * @return int[]
     */
    private static function horasDisponiblesEntreCitas($citas,$estimado){
        $i = 0;
        $horas = [];
        $hora_anterior = '';
        foreach ($citas as $cita){
            if($i != 0){
                $minutos = self::diferenciaHoras($hora_anterior,\date('H:i',strtotime($cita->fecha)));
                if($minutos > $estimado){
                    $horasRelleno = self::rellenarHoras(\date('H',strtotime($hora_anterior)),\date('H',(strtotime($cita->fecha) - ($estimado * 60))));
                    $horas = array_merge($horas,$horasRelleno);
                }
            }
            $i++;
            $hora_anterior =\date('H:i',strtotime($cita->fecha_fin));
        }
        return $horas;
    }
    
    /**
     * 
     * @param string $horaI
     * @param string $horaF
     * @return string[]
     */
    private static function rellenarHoras($horaI,$horaF,$incluisiva = true){
        $horas = [];
        for($i = intval($horaI); ($incluisiva ? $i <= intval($horaF) : $i < intval($horaF)); $i++){
            $horas[] = $i.'';
        }
        return $horas;
    }

    
    private static function obtenerHoraFromString($tiempo){
        return intval(\date('H', strtotime($tiempo)));
    }
    
    private static function obtenerMinutoFromString($tiempo){
        return intval(\date('i', strtotime($tiempo)));
    }
    
    private static function obtenerHoraFromTimeStamp($tiempo){
        return intval(\date('H',$tiempo ));
    }
    
    private static function obtenerMinutoTimeStamp($tiempo){
        return intval(\date('i',$tiempo));
    }

}
