<?php

namespace app\models;

use app\customs\CitaPeriodica;
use Yii;
use app\customs\Utils;
/**
 * This is the model class for table "cita".
 *
 * @property int $id
 * @property int $id_suscripcion
 * @property int $id_clinica
 * @property string $fecha
 * @property int $id_paciente
 * @property int $estado
 * @property int $tiempo_estimado
 * @property string $asunto
 * @property string $diagnostico
 * @property string $receta
 * @property string $duracion
 * @property string $fechaDB
 * @property string $fechaFin
 * @property float $costo
 * @property int $tipo_cita
 * @property Paciente $paciente
 * @property Suscripcion $suscripcion
 * @property Clinica $clinica
 * @property Feedback[] $feedbacks
 * @property ProcedimientoCita[] $procedimientoCitas
 * @property BitacoraCitaMensaje[] $bitacoraCitaMensajes
 * @property int $totalRecordatorios
 * @property BitacoraCitaMensaje $ultimoRecordatorio
 * @property string $textoEstado
 * @property string $nombrePaciente
 * @property string $nombreClinica
 * @property CobroPagadito $cobroPagadito
 * @property string $id_cita_google_calendar
 * @property string $horaFormat
 * @property string $procedimientosAplicadosStr
 * @property string $cita_periodica_token
 * @property bool $esCitaPeriodica
 * @property int $repetirCita
 * @property string $periodo
 * @property int $rango
 * @property int $repeticiones
 * @property int $diaSemana
 * @property string $finalizarRepeticionPor
 * @property BitacoraCita $bitacoraCita
 */
class Cita extends \yii\db\ActiveRecord
{
    const AGENDADA = 0;
    const REPROGRAMADA = 1;
    const CONFIRMADA = 2;
    const CANCELADA = 3;
    const COMPLETADA = 4;
    const INASISTENCIA = 5;
    
    const CITA_PRESENCIAL = 0;
    const CITA_EN_LINEA = 1;
    
    public static $estados = [ 
        -1=>'Todas', 
        0 => 'Agendada', 
        1 => 'Reprogramada', 
        2 => 'Confirmada', 
        3 => 'Cancelada', 
        4 => 'Completada', 
        5 => 'Inasitencia'
    ];
    
    
    public $hora;
    
    public $hasFormat = false;

    /**
     * Repeticion de la cita
    */
    public $repetirCita = CitaPeriodica::NO_REPETIR;
    /**
     * Boleano que establece si es una cita periodica
    */
    public $periodo = CitaPeriodica::SEMANA;
    /**
     * Entero que establece el espacio entre cada cita
    */
    public $rango = 1;
    /**
     * Entero que establece el número de citas périodicas que se van a crear
    */
    public $repeticiones = 4;
    /**
     * Entero que establece el día de la semana escogido para repetir la cita
    */
    public $diaSemanaRepeticion = 1;//Por defecto lunes
    /**
     * Fecha limite de repeticion de citas
    */
    public $fechaLimite = '';
    /**
     * Entero que establece el día de la semana escogido para repetir la cita
    */
    public $finalizarRepeticionPor = CitaPeriodica::FINAL_POR_FECHA;



    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'cita';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id_suscripcion', 'fecha', 'id_paciente', 'estado','tiempo_estimado','id_clinica', 'costo'], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.'],
            [['id_suscripcion', 'id_paciente', 'estado','tiempo_estimado','id_clinica', 'tipo_cita'], 'integer'],
            [['fecha', 'hora'], 'safe'],
            [['costo'], 'number'],
            [['hora'], 'string', 'max' => 8],
            [['asunto', 'diagnostico', 'receta'], 'string', 'max' => 3000],
            [['id_cita_google_calendar'], 'string', 'max' => 45 ],
            [['rango'], 'integer', 'min' => 1],
            [['repetirCita', 'diaSemanaRepeticion'], 'integer'],
            [['fechaLimite', 'finalizarRepeticionPor', 'periodo'], 'string'],
            [['repeticiones'], 'integer', 'min' => 2],
            [['id_paciente'], 'exist', 'skipOnError' => true, 'targetClass' => Paciente::className(), 'targetAttribute' => ['id_paciente' => 'id']],
            [['id_suscripcion'], 'exist', 'skipOnError' => true, 'targetClass' => Suscripcion::className(), 'targetAttribute' => ['id_suscripcion' => 'id']],
            [['id_clinica'], 'exist', 'skipOnError' => true, 'targetClass' => Clinica::className(), 'targetAttribute' => ['id_clinica' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'id_suscripcion' => 'Suscripción',
            'fecha' => 'Fecha',
            'hora' => 'Horas',
            'id_paciente' => 'Paciente',
            'estado' => 'Estado',
            'asunto' => 'Motivo de consulta',
            'diagnostico' => 'Diagnóstico',
            'receta' => 'Receta',
            'tiempo_estimado' => 'Tiempo Estimado (En Minutos)',
            'id_clinica' => 'Clínica',
            'duracion' => 'Duración',
            'procedimientosAplicados' => 'Procedimientos',
            'nombrePaciente' => 'Paciente',
            'nombreClinica' => 'Clínica',
            'textoEstado' => 'Estado',
            'costo' => 'Costo'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPaciente()
    {
        return $this->hasOne(Paciente::className(), ['id' => 'id_paciente']);
    }

    /**
     * @return string
     */
    public function getNombrePaciente()
    {
        return $this->paciente->nombreCompleto;
    }

    /**
     * @return string
     */
    public function getNombreClinica()
    {
        return $this->clinica->nombre;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        return $this->hasOne(Suscripcion::className(), ['id' => 'id_suscripcion']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinica()
    {
        return $this->hasOne(Clinica::className(), ['id' => 'id_clinica']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUrlCita()
    {
        return $this->hasOne(UrlCita::className(), ['id' => 'id_suscripcion']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getProcedimientoCitas()
    {
        return $this->hasMany(ProcedimientoCita::className(), ['id_cita' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getFeedbacks()
    {
        return $this->hasMany(Feedback::className(), ['id_cita' => 'id']);
    }
    
    /**
     * @return string
     */
    public function getDuracion()
    {
        $horas = intval($this->tiempo_estimado/60);
        $horasLabel =  $horas == 1 ? ' hora' : ' horas';
        $tiempoEstimado = $this->tiempo_estimado - ($horas * 60);
        return $horas == 0 ? '' : $horas.$horasLabel.', '.$tiempoEstimado. ' minutos';
    }
        
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat){
                $fecha = $this->fechaDB;
                $this->fecha = $fecha;
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }

    public function getFechaDB(){
        if(!$this->hasFormat){
            return \DateTime::createFromFormat('d/m/Y H:i', $this->fecha)->format('Y-m-d H:i');
        }
        return $this->fecha;
    }

    public function getFechaFin(){
        return \date('Y-m-d H:i',(strtotime($this->fechaDB) + ($this->tiempo_estimado * 60)));
    }
     
    public function afterFind(){
        $fecha = $this->fecha;
        if(!is_null($fecha) && !empty($fecha)){
            $fecha_f = \DateTime::createFromFormat('Y-m-d H:i:s', $fecha)->format('d/m/Y H:i');
            $this->fecha = $fecha_f;
            $this->hasFormat = false;
        }
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBitacoraCitaMensajes()
    {
        return $this->hasMany(BitacoraCitaMensaje::className(), ['id_cita' => 'id']);
    }
    
    public function getHoraFormat() {
        return date(' H:i', strtotime( $this->hora ) );
    }
    
    /**
     * 
     * @return int
     */
    public function getTotalRecordatorios(){
        return $this->getBitacoraCitaMensajes()->count();
    }
    
    /**
     * 
     * @return \app\models\BitacoraCitaMensaje
     */
    public function getUltimoRecordatorio(){
        return $this->getBitacoraCitaMensajes()->orderBy(['fecha' => SORT_DESC])->limit(1)->one();
    }
    
    public function getTextoEstado(){
        return Utils::nombreEstadoCita($this->estado);
    }
        
    /**
     * @return \yii\db\CobroPagadito
     */
    public function getCobroPagadito(){
        return $this->hasOne(CobroPagadito::className(), ['id_cita'=> 'id']);
    }
    
    public function getProcedimientosAplicadosStr(){
        $nombresProcs = \array_map(function($p){ return $p->procedimiento->nombre; }, $this->procedimientoCitas );
        return implode(', ', $nombresProcs);
    }
    /**
     * @return \yii\db\BitacoraCita
     */
    public function getBitacoraCita(){
        return $this->hasOne(BitacoraCita::class, ['id_cita'=> 'id']);
    }
    /**
     * @return bool
     */
    public function getEsCitaPeriodica(){
        return !empty( $this->cita_periodica_token );
    }
    
}
