<?php


namespace app\customs;

use yii\helpers\Url;
use app\models\Paciente;


/**
 * Centralizar la creacion de eventos en Google Calendar 
 *
 * @author <alvaradog.antonio600@gmail.com>
 */
class CalendarGoogle {
    
    
    /**
     * Crear el evento en el calendario de Google y retornar el ID para asociarlo al modelo de la DB.
     * @param  app\models\Cita|app\models\Evento|app\models\RecordatorioDoctor $model Datos del evento
     * @return string
     */
    public static function crearEvento($model){
        $google_calendar = SesionUtils::configuracionSesion('google_calendar');
        $calendarId = $google_calendar->valor;
        if( empty( $calendarId ) ){
            return null;
        }
        $class = self::getModelClassName( $model );//Nombre de la clase del objeto pasado como parametro
        list( $start_date, $end_date  ) = self::getFechaInicioAndFin( $model, $class );
        list( $paciente_nombre, $paciente_email ) = self::getPacienteNameAndEmail( $model, $class );
        $label = self::getLabel( $model, $class );
        $descripcion = self::getDescripcion( $model, $class );

        $username = \Yii::$app->user->identity->suscripcion->email;
        $redirectUrl = Url::to(['/google-api/auth'],true);
        $googleApi = new GoogleCalendarApi($username,$calendarId, $redirectUrl);
        $reminders = $class !== 'RecordatorioDoctor' 
                ? array(
                    'useDefault' => false,
                    'overrides' => array(
                        array('method' => 'popup', 'minutes' => 30),
                        array('method' => 'popup', 'minutes' => 10),
                    ),
                )
                : array();

        if( $googleApi->checkIfCredentialFileExists() ){
            $event = array(
                'summary' => $label,
                'location' => 'San Salvador, El Salvador',
                'description' => $descripcion,
                'start' => array(
                    'dateTime' => $start_date,
                    'timeZone' => 'America/El_Salvador',
                ),
                'end' => array(
                    'dateTime' => $end_date,
                    'timeZone' => 'America/El_Salvador',
                ),
                'recurrence' => array( 'RRULE:FREQ=DAILY;COUNT=1' ),
                'attendees' => array(
                    array( 'displayName' => $paciente_nombre, 'email' => $paciente_email ),                
                ),
                'reminders' => $reminders,
            );
            
            $event['attendeesOmitted'] = empty( $paciente_email ) ? true : false;
            
            $calEvent = $googleApi->createGoogleCalendarEvent($event);
            $getEvent = $googleApi->getGoogleCalendarEvent($calEvent->getId());


            return $getEvent->getId();
        } else {
            return null;
        }
    }
    
    /**
     * Obtener el nombre de la clase del parametro recivido
     * @param Object $model
     * @param string $class
     * @return string
     */
    private static function getModelClassName($model){
        $reflector = new \ReflectionClass( $model );
        return $reflector->getShortName(); 
    }
    
    
    /**
     * Obtener los datos del paciente para asociarlo al evento
     * @param Object $model
     * @param string $class
     */
    private static function getPacienteNameAndEmail($model, $class){
        $paciente_nombre = '';
        $paciente_email = '';
        
        if ( $class == 'Cita' ) {
            $paciente = Paciente::findOne($model->id_paciente);
            $paciente_nombre = $paciente->nombreCorto;
            $paciente_email = $paciente->email;
        } else if( $class == 'RecordatorioDoctor' && !empty( $model->id_paciente ) ){
            $paciente = Paciente::findOne($model->id_paciente);
            $paciente_nombre = $paciente->nombreCorto;
            $paciente_email = $paciente->email;
        }
        return [ $paciente_nombre, $paciente_email ];
    }
    
    /**
     * Obtener la fecha de inicio y fin del evento
     * @param Object $model
     * @param string $class
     * @return array [ $fecha_inicio, $fecha_fin ]
     */
    private static function getFechaInicioAndFin($model, $class){
        $start_date = null;
        $end_date = null;
        $formato = 'd/m/Y H:i';
        $w3c  = "Y-m-d\TH:i:sP";
        if( $class == 'Cita' ){
            $intervalo = new \DateInterval('PT'.$model->tiempo_estimado."M");
            $fecha_hora = $model->fecha . $model->horaFormat;
            $fecha_nueva = \DateTime::createFromFormat($formato, $fecha_hora); 
            $start_date = $fecha_nueva->format($w3c);
            $fecha_final = $fecha_nueva->add($intervalo);
            $end_date   = $fecha_final->format($w3c);
        } else if ( $class == 'RecordatorioDoctor' ){
            $fecha_inicio = $model->todo_dia ? $model->fecha_inicio.' 00:00' : $model->fecha_inicio;
            $intervalo = $model->todo_dia ? new \DateInterval ("PT24H") : new \DateInterval("PT30M");  
            $fecha_inicio_format = \DateTime::createFromFormat($formato, $fecha_inicio); 
            $start_date = $fecha_inicio_format->format($w3c);
            $fecha_fin_format = $fecha_inicio_format->add($intervalo);
            $end_date   = $fecha_fin_format->format($w3c);
        } else if( $class == 'Evento' ){
            $fecha_inicio = $model->todo_dia ? $model->fecha_inicio.' 00:00' : $model->fecha_inicio;
            $intervalo = new \DateInterval ("PT24H");  
            $fecha_inicio_format = \DateTime::createFromFormat($formato, $fecha_inicio); 
            $start_date = $fecha_inicio_format->format($w3c);
            $fecha_fin_format = $model->todo_dia ? $fecha_inicio_format->add($intervalo) : \DateTime::createFromFormat($formato, $model->fecha_fin);
            $end_date   = $fecha_fin_format->format($w3c);
        }
        
        return [ $start_date, $end_date ];
    }
    
    /**
     * Obtener etiqueta del evento a crear 
     * @param Object $model
     * @param string $class
     * @return string Descripcion por evento
     */
    private static function getLabel($model, $class){
        if ( $class == 'Cita' ) {
            return 'Cita - ' . Paciente::findOne($model->id_paciente)->nombreCorto;
        } else if ( $class == 'Evento' ){
            return 'Evento - ' . $model->nombre;
        }
        return 'Recordatorio - ' . $model->asunto;
    }
    
    /**
     * Obtener la descripción del evento
     * @param Object $model
     * @param string $class
     * @return string Descripcion por evento
     */
    private static function getDescripcion($model, $class){
        if ( $class == 'Cita' ) {
            return $model->diagnostico;
        } else if ( $class == 'Evento' ){
            return empty( $model->descripcion ) ? '(Descripción no definida)' : $model->descripcion;
        }
        return empty( $model->comentarios ) ? '(No hay comentarios)' : $model->comentarios;
    }
    
    /**
     * Eliminar un evento del calendario de google
     * @param string $id
     * @return boolean
     */
    public static function eliminarEvento($id) {
        $calendarId = SesionUtils::configuracionSesion('google_calendar')->valor;
        if( empty( $calendarId ) ){
            return false;
        }
        $username = \Yii::$app->user->identity->suscripcion->email;
        $redirectUrl = Url::to(['/google-api/auth'],true);

        $googleApi = new GoogleCalendarApi($username,$calendarId, $redirectUrl);
        
        if( $googleApi->checkIfCredentialFileExists() ){
            $googleApi->deleteGoogleCalendarEvent( $id );
            return true; 
        }else{
            return false;
        }
    }


    /**
     * Actualizar un evento del calendario de google
     * @param string $id
     * @return boolean
     */
    public static function actualizarEvento($model) {
        $calendarId = SesionUtils::configuracionSesion('google_calendar')->valor;
        if( empty( $calendarId ) ){
            return false;
        }
        $username = \Yii::$app->user->identity->suscripcion->email;
        $redirectUrl = Url::to(['/google-api/auth'],true);

        $googleApi = new GoogleCalendarApi($username,$calendarId, $redirectUrl);
        
        if( $googleApi->checkIfCredentialFileExists() ){            
            $id=$model->id_evento_google_calendar;
            $googleApi->updateGoogleCalendarEvent( $id,$model );
            return true; 
        }else{
            return false;
        }
    }
    
}
