<?php

namespace app\controllers;

use Yii;
use app\models\HistorialMedicoPaciente;
use app\models\Receta;
use app\customs\BaseController;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\web\Response;
use yii\helpers\ArrayHelper;
use kartik\mpdf\Pdf;
use app\customs\SesionUtils;

/**
 * ProcedimientoController implements the CRUD actions for HistorialMedicoPaciente model.
 */
class PlantillaDiagnosticoCitaController extends BaseController
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Retorna todos los historiales o por una fecha determinada. Retorna un json para pintar los elemtos en el historial médico.
     * @return mixed
     */
    public function actionHistorialPaciente()
    {        
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $fecha = $post['fecha'] == 'all' ? 'all' : \DateTime::createFromFormat('d-m-Y', $post['fecha'])->format('Y-m-d');
        $id_paciente = $post['id'];
        $historial = ArrayHelper::merge($this->obtenerRecetasPaciente($fecha,$id_paciente), $this->obtenerHistorialMedico($fecha,$id_paciente));
        ArrayHelper::multisort($historial, 'date',SORT_DESC);
        return ['Correcto'=>true, 'Historial'=>$historial];
    }
    
    /*
     * Historiales médicos paciente
     * @param string $fecha
     * @param int $id
     * @return Array  
     */
    private function obtenerHistorialMedico($fecha,$id){
        if($fecha == 'all'){
            $model = HistorialMedicoPaciente::find()
                ->where(['id_paciente' => $id])
                ->orderBy(['fecha'=>SORT_DESC])
                ->all();
        } else {
            $model = HistorialMedicoPaciente::find()
                ->where(['id_paciente' => $id])
                ->andWhere(['fecha' => $fecha])    
                ->orderBy(['fecha' => SORT_DESC])
                ->all();
        }

        $res = [];
        foreach ($model as $d){
            $res[] = [
                'id' => $d->id,
                'tipo' => 'historial',
                'id_paciente' => $d->id_paciente,
                'cuerpo' => $d->formatoCuerpo,
                'fecha' => \DateTime::createFromFormat('Y-m-d', $d->fecha)->format('d/m/Y'),
                'date' => strtotime($d->fecha)    
            ];
        }
        
        return $res;
    }
    
    /*
     * Historiales médicos paciente
     * @param string $fecha
     * @param int $id
     * @return Array  
     */
    private function obtenerRecetasPaciente($fecha,$id){
        if($fecha == 'all'){
            $model = Receta::find()
                ->where(['id_paciente' => $id])
                ->orderBy(['fecha_cita'=>SORT_DESC])
                ->all();
        } else {
            $model = Receta::find()
                ->where(['id_paciente' => $id])
                ->andWhere(['fecha_cita' => $fecha])    
                ->orderBy(['fecha_cita' => SORT_DESC])
                ->all();
        }

        $res = [];
        foreach ($model as $r){
            $res[] = [
                'id' => $r->id,
                'tipo' => 'receta',
                'id_paciente' => $r->id_paciente,
                'cuerpo' => $r->formatoReceta,
                'fecha' => $r->fecha_cita,
                'date' => strtotime(\DateTime::createFromFormat('d/m/Y', $r->fecha_cita)->format('Y-m-d'))    
            ];
        }
        return $res;
    }
    
    /*
     * Para todo el listado de fechas en el grid de historiales y recetas
     * @return mixed
     */
    public function actionHistorialMedicoPacienteFechas($id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $historialesOBJ   = $this->obtenerHistorialMedico('all',$id);
        $recetasOBJ       = $this->obtenerRecetasPaciente('all',$id);
        $historialFechas  = [];
        $historialRecetas = [];
        $fechas = [];
        foreach($historialesOBJ as $h){$historialFechas[] = $h['date'];}
        foreach($recetasOBJ as $r){$historialRecetas[] = $r['date'];}
        if(count($historialFechas) > count($historialRecetas)){
            $fechas = $historialFechas;
            foreach($historialRecetas as $r){
                if(!in_array($r, $fechas)){$fechas[] = $r;}
            }
        } else {
            $fechas = $historialRecetas;
            foreach($historialFechas as $h){
                if(!in_array($h, $fechas)){$fechas[] = $h;}
            }
        }
        $fechas = array_values(array_unique($fechas));
        $fechasOrd = $this->burbujaFechas($fechas,count($fechas));
        return ['Correcto'=>true, 'Historial'=>$fechasOrd];
    }
    
    /*
     * Recibe un array de fechas en formato time '1591056000' para ordenarlos con el metodo burbuja.
     * Después se reconvierte en string de fechas
     * @param array $arr, integer $n
     * @return array 
     */
    private function burbujaFechas($arr,$n){
        $fechas = [];
        for($i = 1; $i<=$n; $i++){
            for($j = 0; $j<$n-$i;$j++){
                if($arr[$j] < $arr[$j+1]){
                    $aux = $arr[$j+1];
                    $arr[$j+1] = $arr[$j];
                    $arr[$j] = $aux;
                }
            }
        }
        foreach($arr as $f){$fechas[] = date('d/m/Y',$f);}
        return $fechas;
    }


    
    /**
     * Displays a single HistorialMedicoPaciente model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $model = $this->findModel($id);
        if($model->paciente->id_suscripcion != $usuario->suscripcion->id){
            return $this->redirect(["site/forbidden"], 403);
        }
        $fecha = \DateTime::createFromFormat('Y-m-d', $model->fecha)->format('d/m/Y');
        return ['Correcto'=>true, 'id' =>$model->id, 'cuerpo'=>$model->cuerpo, 'cuerpoFormat'=>$model->formatoCuerpo, 'fecha'=>$fecha];
    }

    /**
     * Creates a new HistorialMedicoPaciente model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $model = new HistorialMedicoPaciente();
        $usuario = Yii::$app->user->identity;
        $post = Yii::$app->request->post();
        if(Yii::$app->request->isAjax){
            $model->id_paciente = $post['id_paciente'];
            $model->cuerpo = $post['cuerpo'];
            $model->fecha = $post['fecha'];
            if($model->save()){
                return ['Correcto'=>true, 'model' => $model];
            }
            return ['Correcto'=>false];
        }

    }

    /**
     * Updates an existing HistorialMedicoPaciente model.
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $post = Yii::$app->request->post();
        if(Yii::$app->request->isAjax){
            $model = $this->findModel($post['id']);
            if($model->paciente->id_suscripcion != $usuario->suscripcion->id){
                return $this->redirect(["site/forbidden"], 403);
            }
            $model->fecha = $post['fecha'];
            $model->cuerpo = $post['cuerpo'];
            if($model->save()){
                return ['Correcto'=>true, 'model' => $model];
            }
            return ['Correcto'=>false];
        }
        
    }
    
    public function actionHistorialPdf($id)
    {
        Yii::$app->response->format = Response::FORMAT_RAW;
        $model = $this->findModel($id);
        $usuario = \Yii::$app->user->identity;
        $prefijo = SesionUtils::getLlaveValor('identificativo_profesion')->valor;
        $nombrePublico = SesionUtils::getLlaveValor('nombre_publico')->valor;
        if($model->paciente->id_suscripcion != $usuario->suscripcion->id){
            return $this->redirect(["site/forbidden"], 403);
        }
        $pdf = new Pdf([
            'mode'=>'',
            'destination'=> Pdf::DEST_BROWSER,
            'content' => $this->renderPartial('../historial-medico-paciente/historial-pdf', [
                'model'=>$model, 
                'nombre_medico'=> ucfirst($prefijo).'. '.$nombrePublico
            ]),
            'cssFile' => '../web/css/recetaPDF.css',
            'options' => [
            ],
            'methods' => [
                'SetTitle' => 'oliviahealth.com',
                'SetSubject' => 'Historial Médico',
                'SetHeader' => [''],
                'SetAuthor' => 'OliviaHealth',
                'SetCreator' => 'OliviaHealth',
                'SetKeywords' => '',
            ]           
        ]);
        
        return $pdf->render();
        
    }        

    /**
     * Deletes an existing HistorialMedicoPaciente model.
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $model = $this->findModel($post['id']);
        if($model->paciente->id_suscripcion != $usuario->suscripcion->id){
            return ['Correcto'=>false];
        }
        $model->delete();
        return ['Correcto'=>true];
    }
    

    /**
     * Finds the HistorialMedicoPaciente model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return PlantillaDiagnostico the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = HistorialMedicoPaciente::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }
    
        public function excludeFromSecurity(){
         return ['historial-paciente', 'historial-medico-paciente-fechas', 'historial-pdf', 'view', 'create',  'update', 'delete'];
    }
    
    public function beforeAction($action)
    {            
        if ($action->id == 'historial-paciente'|| $action->id == 'create' || $action->id == 'update' || $action->id == 'delete') {
            $this->enableCsrfValidation = false;
        }

        return parent::beforeAction($action);
    }
    
}
