<?php

namespace app\controllers;

use Yii;
use app\models\Rol;
use yii\web\Response;
use app\models\Perfil;
use app\models\Clinica;
use app\models\Usuario;
use yii\widgets\ActiveForm;
use app\models\Controlador;
use yii\filters\VerbFilter;
use yii\helpers\ArrayHelper;
use app\models\GrupoTrabajo;
use app\models\PermisoUsuario;
use app\customs\BaseController;
use yii\data\ActiveDataProvider;
use app\models\UsuarioSuscripcion;
use yii\web\NotFoundHttpException;
use app\models\GrupoTrabajoUsuario;
use app\models\repositories\TokenRepository;
/**
 * AsistenteController implements the CRUD actions for Usuario model.
 */
class ColegaController extends BaseController {

    /**
     * {@inheritdoc}
     */
    public function behaviors() {
        return [
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                    'buscar-asistente' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Lists all Usuario models.
     * @return mixed
     */
    public function actionIndex() {
        $usuario = Yii::$app->user->identity;

        if ($usuario->tipo == Usuario::ASISTENTE || $usuario->tipo == Usuario::ADMIN) {
            return $this->redirect(['site/desautorizado']);
        }
        $tipoJerarquia = $usuario->tipoUsuarioJerarquia;
        $colegaQuery = UsuarioSuscripcion::find()->where(['id_suscripcion' => $usuario->suscripcion->id]);
        if($tipoJerarquia != UsuarioSuscripcion::TIPO_ADMINISTRADOR){
            $grupos = GrupoTrabajoUsuario::find()->where(['id_usuario' => $usuario->id])->select('id_grupo')->column();
            $usuariosGrupo = GrupoTrabajoUsuario::find()->where(['id_grupo' => $grupos])->andWhere(['<>','id_usuario',$usuario->id])->select('id_usuario')->column();
            $colegaQuery->andFilterWhere(['id_usuario' => $usuariosGrupo]);
        }
        
        
        $dataProvider = new ActiveDataProvider([
            'query' => $colegaQuery,
        ]);

        return $this->render('index', [
                    'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Usuario model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id) {
        $usuarioSus = UsuarioSuscripcion::findOne(['id' => $id]);
        $usuario = Yii::$app->user->identity;
        $controladores = Controlador::find()
            ->select(['controlador.id', 'controlador.nombre', 'controlador.nombre_amigable', 'controlador.descripcion'])
            ->innerJoin(['a' => 'accion'],'`controlador`.`id` = `a`.`id_controlador`')
            ->where(['a.tipo' => 1])
            ->groupBy(['controlador.nombre'])
            ->orderBy(['controlador.nombre'=> \SORT_ASC])
            ->all();
        return $this->render('view', [
                    'model' => $usuarioSus->usuario,
                    'usuarioSuscripcion' => $usuarioSus,
                    'controladores'=>$controladores,
                    'idSuscripcion' => $usuario->suscripcion->id
        ]);
    }
    
   
    
    /**
     * Notifica la existencia de un usuario al crear un asistente
     * @return mixed
     */
    public function actionExisteUsuario($email){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Usuario::find()->where(['nombre_usuario' => $email])->andFilterWhere(['activo' => true])->one();
        if($usuario != null){
            return ['Correcto'=>true, 'Existe'=>true];
        }
        return ['Correcto'=>true, 'Existe'=>false];
    }

    /**
     * Creates a new Usuario model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate() {
        $usuario = Yii::$app->user->identity;
        $idSuscripcion = $usuario->suscripcion->id;
        $model = new Usuario();
        $perfil = new Perfil();
        $post = Yii::$app->request->post();
        $usuarioSuscripcion = new UsuarioSuscripcion();
        $usuarioSuscripcion->tipo = UsuarioSuscripcion::TIPO_ASISTENTE;
        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        } else if ($model->load($post) && $perfil->load($post)) {
            $tipoUsuarioSuscripcion = $post['tipo'];
            $model->activo = 1;
            $model->tipo =  $tipoUsuarioSuscripcion == UsuarioSuscripcion::TIPO_ASISTENTE ? Usuario::ASISTENTE : Usuario::DOCTOR;
            $model->setValidateOldPassword(false);
            $password = TokenRepository::getToken(TokenRepository::TamanioPassword);
            $model->password = $password;
            $model->passwordConfirm = $password;
            $model->oldPassword = "123456";
            if ($model->save()) {
                $perfil->email = $model->nombre_usuario;
                $perfil->id = $model->id;
                if ($perfil->save()) {
                    $usuarioSuscripcion->activo = 1;
                    if($tipoUsuarioSuscripcion != UsuarioSuscripcion::TIPO_ASISTENTE){
                        $usuarioSuscripcion->id_clinica = $this->crearClinica($perfil,$idSuscripcion);
                    }
                    $usuarioSuscripcion->id_suscripcion = $idSuscripcion;
                    $usuarioSuscripcion->id_usuario = $model->id;
                    $usuarioSuscripcion->tipo = $tipoUsuarioSuscripcion;
                    $usuarioSuscripcion->save();
                    
                    $grupoPost = $post['grupo'];
                    $grupos = (is_array($grupoPost)) ? $grupoPost : [$grupoPost];
                    foreach($grupos as $grupo) {
                        $usuarioGrupo = new GrupoTrabajoUsuario();
                        $usuarioGrupo->id_grupo = $grupo;
                        $usuarioGrupo->id_usuario = $model->id;
                        $usuarioGrupo->save();
                    }
                    
                    //TODO: Habilitar el envío de password
                    $this->enviarPasswordColega($model, $password);
                    
                    Yii::$app->session->set("usuarios_clinica_jerarquia",null);
                    Yii::$app->session->set("usuarios_clinica",null);
                    Yii::$app->session->set("clinica_seleccionada",null);
                    
                    return $this->redirect(['index']);
                }
            }
        }
        $gruposQuery = GrupoTrabajo::find()->where(['id_suscripcion' => $idSuscripcion])->orderBy("nombre");
        if($usuario->tipoUsuarioJerarquia == UsuarioSuscripcion::TIPO_COORDINADOR){
            $gruposPermitidos = $usuario->getGruposTrabajoUsuario()->select('id_grupo')->column();
            $gruposQuery->andFilterWhere(['id' => $gruposPermitidos]);
        }
        
        $grupos = ArrayHelper::map($gruposQuery->all(),'id','nombre');
        $rol = Rol::findOne(['llave' => 'asistente_jerarquia']);
        $model->id_rol = $rol->id;
        return $this->render('create', [
                    'model' => $model,
                    'perfil' => $perfil,
                    'grupos' => $grupos,
                    'usuarioSuscripcion' => $usuarioSuscripcion
                ]);
    }
    
    /**-
     * Updates an existing Usuario model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id) {
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $usuarioSuscripcion = UsuarioSuscripcion::findOne(['id' => $id]);
        $model = $usuarioSuscripcion->usuario;
        $perfil = $model->perfil;
        $post = Yii::$app->request->post();
        if ($perfil->load($post) && $perfil->save()) {
            
            $usuarioSuscripcion->tipo = $post['tipo'];
            if($usuarioSuscripcion->save()){
                $gruposUsuario = GrupoTrabajoUsuario::deleteAll(['id_usuario' => $model->id]);
                $grupoPost = $post['grupo'];
                $grupos = (is_array($grupoPost)) ? $grupoPost : [$grupoPost];
                foreach($grupos as $grupo) {
                    $usuarioGrupo = new GrupoTrabajoUsuario();
                    $usuarioGrupo->id_grupo = $grupo;
                    $usuarioGrupo->id_usuario = $model->id;
                    $usuarioGrupo->save();
                }
                return $this->redirect(['index']);
            }
        }
        
        $gruposQuery = GrupoTrabajo::find()->where(['id_suscripcion' => $suscripcion->id])->orderBy("nombre");
        if($usuario->tipoUsuarioJerarquia == UsuarioSuscripcion::TIPO_COORDINADOR){
            $gruposPermitidos = $usuario->getGruposTrabajoUsuario()->select('id_grupo')->column();
            $gruposQuery->andFilterWhere(['id' => $gruposPermitidos]);
        }
        $grupos = ArrayHelper::map($gruposQuery->all(),'id','nombre');
        
        return $this->render('update', [
                'model' => $model,
                'perfil' => $perfil,
                'grupos' => $grupos,
                'usuarioSuscripcion' => $usuarioSuscripcion
            ]);
    }
    
    
    /**
     * 
     * @param Perfil $perfil
     * @param int $id_suscripcion
     */
    private function crearClinica($perfil, $id_suscripcion){
        $clinica = new Clinica();
        $clinica->nombre = $perfil->nombreCompleto;
        $clinica->id_suscripcion = $id_suscripcion;
        $clinica->save();
        return $clinica->id;
    }
    

    /**
     * Deletes an existing Usuario model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id) {
        $model = UsuarioSuscripcion::findOne(['id' => $id]);
        if ($model->usuario->getBitacoraCitas()->count() > 0) {
            Yii::$app->session->setFlash("danger", "El asistente tiene citas creadas, no puede ser eliminado. Puede ponerlo como inactivo para que no ingrese al sistema");
            return $this->redirect(['index']);
        }
        $id_usuario = $model->id_usuario;
        $model->delete();
        Perfil::deleteAll(['id' => $id_usuario]);
        Usuario::deleteAll(['id' => $id_usuario]);
        
        Yii::$app->session->set("usuarios_clinica_jerarquia",null);
        Yii::$app->session->set("usuarios_clinica",null);
        Yii::$app->session->set("clinica_seleccionada",null);
        
        return $this->redirect(['index']);
    }
    
    

    /**
     * 
     * @param app\models\Usuario $usuario
     */
    private function enviarPasswordColega($usuario, $password) {
        $suscripcion = Yii::$app->user->identity->suscripcion;
        \yii::$app->mailer->compose('email-colega', ['model' => $usuario, 'password' => $password, 'suscripcion' => $suscripcion])
                ->setFrom(['info@oliviahealth.com' => 'Olivia Health'])
                ->setTo($usuario->perfil->email)
                ->setSubject("Se le ha asignado un usuario para Olivia Health")
                ->send();
    }
    
    /**
     * Update password an existing Usuario model.
     * If updation is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdatePassword($id) {
        $model = UsuarioSuscripcion::findOne(['id' => $id]);
        $usuario = $model->usuario;
        $password = TokenRepository::getToken(TokenRepository::TamanioPassword);
        $usuario->password = $password;
        $usuario->passwordConfirm = $password;
        if ($usuario->update()) {
            $this->actualizarPasswordAssistente($usuario, $password);
            \Yii::$app->session->setFlash("success","Contraseña reestablecida correctamente");
            return $this->redirect(['view', 'id'=>$id]);
        }
    }

    /**
     * 
     * @param app\models\Usuario $usuario
     */
    private function actualizarPasswordAssistente($usuario, $password) {
        $suscripcion = Yii::$app->user->identity->suscripcion;
        \yii::$app->mailer->compose('new-password-asistente', ['model' => $usuario, 'password' => $password, 'suscripcion' => $suscripcion])
                ->setFrom(['info@oliviahealth.com' => 'Olivia Health'])
                ->setTo($usuario->perfil->email)
                ->setSubject("Se ha reestablecido tu contraseña para ingresar a Olivia Health")
                ->send();
    }
    
    
     /**
     * @param array $acciones
     * @param app\models\UsuarioSuscripcion $usuarioSuscripcion
     * @param int $idSuscripcion
     * @return boolean
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function guardarPermisosAsistente($acciones, $usuarioSuscripcion,$idSuscripcion){
        PermisoUsuario::deleteAll(['id_suscripcion'=>$usuarioSuscripcion->id_clinica, 'id_usuario'=>$usuarioSuscripcion->id_usuario]);
        foreach ($acciones as $accion){
            $model = new PermisoUsuario;
            $model->id_suscripcion = $idSuscripcion;
            $model->id_usuario = $usuarioSuscripcion->id_usuario;
            $model->id_accion = $accion;
            $model->permitido = 1;
            $model->save();
            unset($model);
        }
        return true;
    }
    
    protected function toSimpleArray($permisos){
        $final=[];
        foreach ($permisos as $permiso){
            array_push($final,$permiso->id_accion);
        }
        return $final;
    }

    /**
     * Finds the Usuario model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Usuario the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id) {
        if (($model = Usuario::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }
       
    public function excludeFromSecurity() {
        return ['buscar-asistente', 'existe-usuario'];
    }
    
    public function checkForSubscription() {
        return ['asociar','buscar-asistente','create', 'update'];
    }
}
