<?php

namespace app\controllers;

use Yii;
use app\models\UsuarioSuscripcion;
use yii\data\ActiveDataProvider;
use app\customs\BaseController;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use app\models\Usuario;
use app\models\Perfil;
use yii\helpers\ArrayHelper;
use app\models\Clinica;
use app\models\repositories\TokenRepository;
use yii\web\Response;
use yii\widgets\ActiveForm;
use app\models\Controlador;
use app\models\PermisoUsuario;
/**
 * AsistenteController implements the CRUD actions for Usuario model.
 */
class AsistenteController extends BaseController {

    /**
     * {@inheritdoc}
     */
    public function behaviors() {
        return [
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                    'buscar-asistente' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Lists all Usuario models.
     * @return mixed
     */
    public function actionIndex() {
        $usuario = Yii::$app->user->identity;

        if ($usuario->tipo == Usuario::ASISTENTE || $usuario->tipo == Usuario::ADMIN) {
            return $this->redirect(['site/desautorizado']);
        }
        $dataProvider = new ActiveDataProvider([
            'query' => UsuarioSuscripcion::find()->where(['id_suscripcion' => $usuario->suscripcion->id]),
        ]);

        return $this->render('index', [
                    'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Usuario model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id) {
        $usuarioSus = UsuarioSuscripcion::findOne(['id' => $id]);
        $usuario = Yii::$app->user->identity;
        $controladores = Controlador::find()
            ->select(['controlador.id', 'controlador.nombre', 'controlador.nombre_amigable', 'controlador.descripcion'])
            ->innerJoin(['a' => 'accion'],'`controlador`.`id` = `a`.`id_controlador`')
            ->where(['a.tipo' => 1])
            ->groupBy(['controlador.nombre'])
            ->orderBy(['controlador.nombre'=> \SORT_ASC])
            ->all();
        return $this->render('view', [
                    'model' => $usuarioSus->usuario,
                    'usuarioSuscripcion' => $usuarioSus,
                    'controladores'=>$controladores,
                    'idSuscripcion' => $usuario->suscripcion->id
        ]);
    }
    
    /**
     * Muestra la información del usuario buscado y encontrado.
     * @return mixed
     */
    public function actionAsociar() {
        $id_usuario = \Yii::$app->session->getFlash("id_usuario_asociar");
        $usuario = Yii::$app->user->identity;
        $id_suscripcion = $usuario->suscripcion->id;
        $post = \Yii::$app->request->post();
        if(isset($id_usuario)){
            $usuarioBuscado = Usuario::findOne(['id' => $id_usuario]);
            $clinicas = Clinica::find()->where(['id_suscripcion' => $id_suscripcion])->all();
            return $this->render('asociar',[
                "model" => $usuarioBuscado,
                'clinicas' => ArrayHelper::map($clinicas, 'id', 'nombre') ]);
        }
        if(isset($post['id_usuario']) && isset($post['id_clinica'])){
            $id_usuario = $post['id_usuario'];
            $id_clinica = $post['id_clinica'];
            $asistente = new UsuarioSuscripcion();
            $asistente->id_suscripcion = $id_suscripcion;
            $asistente->id_clinica = $id_clinica;
            $asistente->id_usuario = $id_usuario;
            $asistente->activo = TRUE;
            $asistente->save();
            Yii::$app->session->setFlash("info","Se ha asociado el asistente.");
        }
        return $this->redirect(['index']);
    }
    
    /**
     * Busca un asistente mediante correo electrónico. Accesible mediante metodo post.
     * @return mixed
     */
    public function actionBuscarAsistente(){
        $id_suscripcion = Yii::$app->user->identity->suscripcion->id;
        $post = \Yii::$app->request->post();
        $email = $post['email'];
        Yii::$app->response->format = Response::FORMAT_JSON;
        if(isset($email)){
            $usuario = Usuario::find()->where(['nombre_usuario' => $email])->andFilterWhere(['activo' => true])->one();
            if($usuario != null){
                $usuario_suscripcion = UsuarioSuscripcion::find()->where(['id_suscripcion'=>$id_suscripcion, 'id_usuario'=>$usuario->id])->one();
                if($usuario_suscripcion != null){
                    return ['Correcto'=>TRUE, 'Encontrado' => FALSE, 'Mensaje' => "El usuario ya esta asociado a su suscripción"];
                }
                \Yii::$app->session->setFlash("id_usuario_asociar", $usuario->id);
                return ['Correcto'=>TRUE,'Encontrado' => TRUE];
            }
            return ['Correcto'=>TRUE,'Encontrado' => FALSE,'Mensaje' => "No se ha encontrado al usuario, o revise que se encuentra activo."];
        }
        return ['Correcto'=>TRUE,'Mensaje' => "No se ha ingresado ningún correo."];
    }
    
    /**
     * Notifica la existencia de un usuario al crear un asistente
     * @return mixed
     */
    public function actionExisteUsuario($email){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Usuario::find()->where(['nombre_usuario' => $email])->andFilterWhere(['activo' => true])->one();
        if($usuario != null){
            return ['Correcto'=>true, 'Existe'=>true];
        }
        return ['Correcto'=>true, 'Existe'=>false];
    }

    /**
     * Creates a new Usuario model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate() {
        $usuario = Yii::$app->user->identity;
        $model = new Usuario();
        $perfil = new Perfil();
        $post = Yii::$app->request->post();
        $usuarioSuscripcion = new UsuarioSuscripcion();
        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        } else if ($model->load($post) && $perfil->load($post)) {
            $perfil->email = $model->nombre_usuario;
            $model->activo = 1;
            $model->tipo = Usuario::ASISTENTE;
            $model->id_rol = \Yii::$app->params['idRolAsistente'];
            $model->setValidateOldPassword(false);
            $password = TokenRepository::getToken(TokenRepository::TamanioPassword);
            $model->password = $password;
            $model->passwordConfirm = $password;
            $model->oldPassword = "123456";
            if ($model->save()) {
                $perfil->id = $model->id;
                if ($perfil->save()) {
                    $usuarioSuscripcion->activo = 1;
                    $usuarioSuscripcion->id_clinica = $post['id_clinica'];
                    $usuarioSuscripcion->id_suscripcion = $usuario->suscripcion->id;
                    $usuarioSuscripcion->id_usuario = $model->id;
                    $usuarioSuscripcion->save();
                    if(isset($post['permisos'])){
                        $permisos = $post['permisos'];
                        $this->guardarPermisosAsistente($permisos, $usuarioSuscripcion, $usuario->suscripcion->id);
                    }
                    $this->enviarPasswordAssistente($model, $password);
                    return $this->redirect(['index']);
                }
            }
        }
        
        return $this->render('create', $this->parametrosVista([
                        'model' => $model,
                        'perfil' => $perfil,
                        'idClinica' => $usuarioSuscripcion->id_clinica,
                        'idSuscripcion' => 0,
                        'idUsuario' => 0
                    ],$usuario,['id_usuario'=> null],$usuarioSuscripcion));
    }
    
    /**
     * Updates an existing Usuario model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id) {
        $usuarioSuscripcion = UsuarioSuscripcion::findOne(['id' => $id]);
        $model = $usuarioSuscripcion->usuario;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $perfil = $model->perfil;
        $clinicas = ArrayHelper::map(Clinica::find()->where(['id_suscripcion' => $usuario->suscripcion->id])->all(), 'id', 'nombre');
        if ($perfil->load(Yii::$app->request->post()) && $perfil->save()) {
            $usuarioSuscripcion->id_clinica = $post['id_clinica'];
            if($usuarioSuscripcion->save()){
                $permisos = isset($post['permisos']) ? $post['permisos'] : [] ;
                $this->guardarPermisosAsistente($permisos, $usuarioSuscripcion, $usuario->suscripcion->id);
              return $this->redirect(['index']);
            }
        }
        $parametrosAccion = [
                    'model' => $model,
                    'perfil' => $perfil,                    
                    'idClinica' => $usuarioSuscripcion->id_clinica,
                    'id_us_suscripcion' => $usuarioSuscripcion->id,
                    'idUsuario' => $usuarioSuscripcion->id_usuario,
                    'idSuscripcion' => $usuario->suscripcion->id

        ];
        return $this->render('update', $this->parametrosVista($parametrosAccion, $usuario,['id_usuario'=> $usuarioSuscripcion->id_usuario, 'id_suscripcion'=>$usuarioSuscripcion->id_clinica]));
    }
    
    /*
     * Recibir parametros de vista update y create retornar array connuevos parametros
     */
    private function parametrosVista($parametrosAccion, $usuario, $wherePermisos){
        $clinicas = ArrayHelper::map(Clinica::find()->where(['id_suscripcion' => $usuario->suscripcion->id])->all(), 'id', 'nombre');
        $controladores = Controlador::find()
            ->select(['controlador.id', 'controlador.nombre', 'controlador.nombre_amigable', 'controlador.descripcion'])
            ->innerJoin(['a' => 'accion'],'`controlador`.`id` = `a`.`id_controlador`')
            ->where(['a.tipo' => 1])
            ->groupBy(['controlador.nombre'])
            ->orderBy(['controlador.nombre'=> \SORT_ASC])
            ->all();
        $permisos= PermisoUsuario::find()
                ->andWhere($wherePermisos)
                ->andWhere(['permitido' => 1])
                ->all();
        $concedidos=$this->toSimpleArray($permisos);
        $parametrosAccion['controladores'] = $controladores;
        $parametrosAccion['concedidos'] = $concedidos;       
        $parametrosAccion['permisos'] = $permisos;
        $parametrosAccion['clinicas'] = $clinicas;
        return $parametrosAccion;

    }

    /**
     * Deletes an existing Usuario model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id) {
        $model = UsuarioSuscripcion::findOne(['id' => $id]);
        if ($model->usuario->getBitacoraCitas()->count() > 0) {
            Yii::$app->session->setFlash("danger", "El asistente tiene citas creadas, no puede ser eliminado. Puede ponerlo como inactivo para que no ingrese al sistema");
            return $this->redirect(['index']);
        }
        $id_usuario = $model->id_usuario;
        $model->delete();
        Perfil::deleteAll(['id' => $id_usuario]);
        Usuario::deleteAll(['id' => $id_usuario]);
        return $this->redirect(['index']);
    }
    
    
    /**
     * Deletes an existing Usuario model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionActivar($id) {
        $model = UsuarioSuscripcion::findOne(['id' => $id]);
        $model->activo = !$model->activo;
        $model->save();
        \Yii::$app->session->setFlash("success","El usuario ha sido " . ($model->activo ? "activado" : "inactivado"));
        return $this->redirect(['index']);
    }

    /**
     * 
     * @param app\models\Usuario $usuario
     */
    private function enviarPasswordAssistente($usuario, $password) {
        $suscripcion = Yii::$app->user->identity->suscripcion;
        \yii::$app->mailer->compose('email-asistente', ['model' => $usuario, 'password' => $password, 'suscripcion' => $suscripcion])
                ->setFrom(['info@oliviahealth.com' => 'Olivia Health'])
                ->setTo($usuario->perfil->email)
                ->setSubject("Se le ha asignado un usuario para Olivia Health")
                ->send();
    }
    
    /**
     * 
     * @param app\models\Usuario $usuario
     */
    private function notificarAsociacionAssistente($usuario) {
        $suscripcion = Yii::$app->user->identity->suscripcion;
        \yii::$app->mailer->compose('asociacion-asistente', ['model' => $usuario, 'suscripcion' => $suscripcion])
                ->setFrom(['info@oliviahealth.com' => 'Olivia Health'])
                ->setTo($usuario->perfil->email)
                ->setSubject("Se le ha asignado un usuario para Olivia Health")
                ->send();
    }

    /**
     * Update password an existing Usuario model.
     * If updation is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdatepassword($id) {
        $model = UsuarioSuscripcion::findOne(['id' => $id]);
        $usuario = $model->usuario;
        $password = TokenRepository::getToken(TokenRepository::TamanioPassword);
        $usuario->password = $password;
        $usuario->passwordConfirm = $password;
        if ($usuario->update()) {
            $this->actualizarPasswordAssistente($usuario, $password);
            \Yii::$app->session->setFlash("success","Contraseña reestablecida correctamente");
            return $this->redirect(['view', 'id'=>$id]);
        }
    }

    /**
     * 
     * @param app\models\Usuario $usuario
     */
    private function actualizarPasswordAssistente($usuario, $password) {
        $suscripcion = Yii::$app->user->identity->suscripcion;
        \yii::$app->mailer->compose('new-password-asistente', ['model' => $usuario, 'password' => $password, 'suscripcion' => $suscripcion])
                ->setFrom(['info@oliviahealth.com' => 'Olivia Health'])
                ->setTo($usuario->perfil->email)
                ->setSubject("Se ha reestablecido tu contraseña para ingresar a Olivia Health")
                ->send();
    }
    
    
     /**
     * @param array $acciones
     * @param app\models\UsuarioSuscripcion $usuarioSuscripcion
     * @param int $idSuscripcion
     * @return boolean
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function guardarPermisosAsistente($acciones, $usuarioSuscripcion,$idSuscripcion){
        PermisoUsuario::deleteAll(['id_suscripcion'=>$usuarioSuscripcion->id_clinica, 'id_usuario'=>$usuarioSuscripcion->id_usuario]);
        foreach ($acciones as $accion){
            $model = new PermisoUsuario;
            $model->id_suscripcion = $idSuscripcion;
            $model->id_usuario = $usuarioSuscripcion->id_usuario;
            $model->id_accion = $accion;
            $model->permitido = 1;
            $model->save();
            unset($model);
        }
        return true;
    }
    
    protected function toSimpleArray($permisos){
        $final=[];
        foreach ($permisos as $permiso){
            array_push($final,$permiso->id_accion);
        }
        return $final;
    }

    /**
     * Finds the Usuario model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Usuario the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id) {
        if (($model = Usuario::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }
       
    public function excludeFromSecurity() {
        return ['buscar-asistente', 'existe-usuario'];
    }
    
    public function checkForSubscription() {
        return ['asociar','buscar-asistente','create', 'update'];
    }
}
