<?php

namespace app\models;

use Yii;
use app\customs\SesionUtils;

/**
 * This is the model class for table "usuario".
 *
 * @property int $id
 * @property string $nombre_usuario
 * @property string $password
 * @property string $llave_password
 * @property int $activo
 * @property int $id_rol
 * @property int $tipo
 *
 * @property Bitacora[] $bitacoras
 * @property Perfil $perfil
 * @property Suscripcion[] $suscripciones
 * @property Rol $rol
 * @property Suscripcion $suscripcion
 * @property UsuarioSuscripcion[] $usuarioSuscripciones
 * @property UsuarioSuscripcion $usuarioSuscripcion
 * @property bool $suscripcionActiva
 * @property FormaPago $formaPago
 * @property int $tipoUsuarioJerarquia
 * @property GrupoTrabajoUsuario $gruposTrabajoUsuario
 * @property GrupoTrabajo[] $gruposTrabajo
 */
class Usuario extends \yii\db\ActiveRecord implements \yii\web\IdentityInterface
{
    const ADMIN = 0;
    const DOCTOR = 1;
    const ASISTENTE = 2;

    public $userName="";
    public $oldPassword;
    public $password;
    public $passwordConfirm;
    private $validateOldPassword=true;
    
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'usuario';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['nombre_usuario', 'password', 'activo', 'id_rol', 'tipo'], 'required','message'=>'El campo \'{attribute}\' no puede quedar vacío.'],
            [['activo', 'id_rol', 'tipo'], 'integer'],
            [['nombre_usuario'], 'string', 'max' => 75],
            [['llave_password'], 'string', 'max' => 300],
            [['id_rol'], 'exist', 'skipOnError' => true, 'targetClass' => Rol::className(), 'targetAttribute' => ['id_rol' => 'id']],
            ['passwordConfirm', 'compare', 'compareAttribute' => 'password','message' => 'Las contraseñas deben ser iguales'],
            ['passwordConfirm', 'passwordConfirmRequired','skipOnEmpty' => false, 'skipOnError' => false],
            ['oldPassword','checkOldPassword'],
            [['nombre_usuario'],'unique', 'message'=> 'El correo ya esta asignado, por favor intente con otro.11']
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'Id',
            'nombre_usuario' => 'Nombre de usuario',
            'password' => 'Password',
            'llave_password' => 'Llave password',
            'activo' => 'Activo',
            'id_rol' => 'Rol',
            'tipo' => 'Tipo',
        ];
    }
    
    public function passwordConfirmRequired($attribute, $params, $validator){
        if($this->$attribute == null || $this->$attribute == ''){
            $this->addError($attribute, 'El campo \'Confirmación del password\' no puede quedar vacío.');
        }
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBitacoras()
    {
        return $this->hasMany(Bitacora::className(), ['id_usuario' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPerfil()
    {
        return $this->hasOne(Perfil::className(), ['id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripciones()
    {
        return $this->hasMany(Suscripcion::className(), ['id_usuario' => 'id']);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        if($this->tipo == self::ADMIN) return null;
        if($this->tipo == self::DOCTOR){
            $suscripcionQuery = $this->hasOne(Suscripcion::className(),['id_usuario' => 'id']);
            if($suscripcionQuery->exists()){
                return $suscripcionQuery;
            }
            else{
                $usuarioSuscripcion = $this->getUsuarioSuscripciones()->where(['activo' => 1])->limit(1)->one();
                return $usuarioSuscripcion->getSuscripcion();
            }
            
        }
        $clinica = SesionUtils::ClinicaSeleccionada();
        return  $clinica->suscripcion;
    }
    
    public function getSuscripcionActiva()
    {
        
        
        if($this->tipo == self::DOCTOR){
            return $this->suscripcion->activa;
        }
        
        return $this->getUsuarioSuscripciones()->where(['activo' => 1])->exists();
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRol()
    {
        return $this->hasOne(Rol::className(), ['id' => 'id_rol']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUsuarioSuscripciones()
    {
        return $this->hasMany(UsuarioSuscripcion::className(), ['id_usuario' => 'id']);
    }
    
    /**
     * 
     * @param int $idSuscripcion
     * @return app\models\Clinica[]
     */
    public function getClinicasActivasSuscripcion($idSuscripcion)
    {
        if($this->tipo == self::ASISTENTE){
            $uSuscripciones = $this->getUsuarioSuscripciones()->where(['activo' => true])->andFilterWhere(['id_suscripcion' => $idSuscripcion])->all();
            $ids = [];
            foreach ($uSuscripciones as $us){
                $ids = $us->id_clinica;
            }
            return Clinica::find()->where(['in','id',$ids])->all();
        }
        return [];
    }
    
    

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUsuarioSuscripcion()
    {
        return $this->hasOne(UsuarioSuscripcion::className(), ['id_usuario' => 'id'])->onCondition(['activo' => 1]);
    }
    
    public function getAuthKey() {
        return $this->llave_password;
        
    }

    public function getId() {
        return $this->id;
    }

    public function validateAuthKey($authKey) {
        return $this->getAuthKey() === $authKey;
        
    }

    public static function findIdentity($id) {
        return static::findOne($id);
    }

    public static function findByUsername($userName) {
        return static::find()->where(['nombre_usuario' => $userName])->one();
    }

    public static function findIdentityByAccessToken($token, $type = null) {
        
    }

    public function validatePassword($passwordToValidate)
    {
        $llave = $this->llave_password;
        $isValid=\Yii::$app->security->validatePassword($passwordToValidate,$llave);
        return $isValid;
    } 
        
    
    public function getUserName(){
        return $this->nombre_usuario;
    }
    
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->llave_password = \Yii::$app->security->generatePasswordHash($this->password);
            }
            else{
                if(!$this->validatePassword($this->password)){
                    $this->llave_password = \Yii::$app->security->generatePasswordHash($this->password);    
                }
            }
            return true;
        }
        return false;
    }
    
    public function checkOldPassword($attribute, $params)
    {
        // no real check at the moment to be sure that the error is triggered
        if ($this->validateOldPassword) {
            if(!$this->validatePassword($this->oldPassword)){
                $this->addError($attribute, 'El password no coincide con el anterior.');
            }
        }
    }
    
    public function setValidateOldPassword($validate)
    {
        $this->validateOldPassword=$validate;
    }
    
    public function login()
    {   
        Yii::$app->user->enableSession = true;
        return Yii::$app->user->login($this, 18000);
    }
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBitacoraCitas()
    {
        return $this->hasMany(BitacoraCita::className(), ['id_usuario' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBitacoraMensajeLibres()
    {
        return $this->hasMany(BitacoraMensajeLibre::className(), ['id_usuario' => 'id']);
    }
    
    public function getFormaPago()
    {
        return $this->hasOne(FormaPago::className(), ['id' => 'id']);
    }
    
    /**
     * Devuelve el tipo de usuario, según la jerarquia
     * @return int
     */
    public function getTipoUsuarioJerarquia(){
        if($this->suscripcion->tipo == Suscripcion::TIPO_INDIVIDUAL){
            throw new Exception('La suscripción no es de jerarquias');
        }
        $usuarioSuscripcion = $this->usuarioSuscripcion;
        return $usuarioSuscripcion == null ? UsuarioSuscripcion::TIPO_ADMINISTRADOR : $usuarioSuscripcion->tipo;
    }
    
    
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getGruposTrabajoUsuario(){
        return $this->hasMany(GrupoTrabajoUsuario::className(), ['id_usuario' => 'id']);
    }
    
    public function getGruposTrabajo(){
        $gruposUsuario = $this->gruposTrabajoUsuario;
        $grupos = [];
        foreach ($gruposUsuario as $grupoUsuario){
            $grupos [] = $grupoUsuario->grupo;
        }
        return $grupos;
    }
}
