<?php

namespace app\models;

use Yii;
use app\customs\Utils;

/**
 * This is the model class for table "evento".
 *
 * @property int $id
 * @property string $nombre
 * @property string $descripcion
 * @property string $fecha_inicio
 * @property string $fecha_fin
 * @property string $fechaInicioDB
 * @property string $fechaFinDB
 * @property int $id_suscripcion
 * @property int $horas
 * @property int $privacidad
 * @property bool $activo
 * @property bool $todo_dia
 * @property string $nombreEvento
 * @property string $descripcionEvento
 * @property string $fechaFinEvento
 * @property int $id_clinica
 * @property string $id_evento_google_calendar
 *
 * @property Suscripcion $suscripcion
 * @property Clinica $clinica
 */
class Evento extends \yii\db\ActiveRecord
{
    public $hasFormat = false;
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'evento';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['nombre', 'id_suscripcion','privacidad','fecha_inicio'], 'required'],
            [['fecha_inicio','fecha_fin'], 'safe'],
            [['id_suscripcion','privacidad','id_clinica'], 'integer'],
            [['activo','todo_dia'], 'boolean'],
            [['nombre'], 'string', 'max' => 200],
            [['descripcion'], 'string', 'max' => 500],
            [['id_evento_google_calendar'], 'string', 'max' => 100],
            [['id_suscripcion'], 'exist', 'skipOnError' => true, 'targetClass' => Suscripcion::className(), 'targetAttribute' => ['id_suscripcion' => 'id']],
            [['fecha_fin'],'compareDates']
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'nombre' => 'Nombre',
            'descripcion' => 'Descripcion',
            'fecha_inicio' => 'Fecha inicio',
            'fecha_fin' => 'Fecha fin',
            'id_suscripcion' => 'Id Suscripcion',
            'activo' => 'Activo',
            'privacidad' => 'Es privado',
            'todo_dia' => 'Todo el día',
            'id_evento_google_calendar' => 'ID Google Calendar'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSuscripcion()
    {
        return $this->hasOne(Suscripcion::className(), ['id' => 'id_suscripcion']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getClinica()
    {
        return $this->hasOne(Clinica::className(), ['id' => 'id_clinica']);
    }
    
    public function beforeSave($insert) {
        if(parent::beforeSave($insert)){
            if(!$this->hasFormat){
                $fechai = \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_inicio . ($this->todo_dia ? ' 00:00' : ''))
                        ->format('Y-m-d H:i');
                if($this->todo_dia){
                    $this->fecha_fin = \date('d/m/Y H:i', strtotime(\date('Y-m-d', strtotime($fechai)) . ' 23:59'));
                }
                $fechaf = \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_fin)->format('Y-m-d H:i');
                $this->fecha_inicio = $fechai;
                $this->fecha_fin = $fechaf;
                $this->hasFormat = true;
            }
            return true;
        }
        return false;
    }
    
    public function afterFind(){
        $fechai = $this->fecha_inicio;
        $fechaf = $this->fecha_fin;
        if(!is_null($fechai) && !empty($fechai) && !is_null($fechaf) && !empty($fechaf)){
            $fecha_i = \DateTime::createFromFormat('Y-m-d H:i:s', $fechai)->format('d/m/Y H:i');
            $this->fecha_inicio = $fecha_i;
            $fecha_f = \DateTime::createFromFormat('Y-m-d H:i:s', $fechaf)->format('d/m/Y H:i');
            $this->fecha_fin = $fecha_f;
            $this->hasFormat = false;
        }
    }
    
    public function compareDates()
    {
        if(!$this->todo_dia){
            $start_date = $this->fecha_inicio;
            if(empty($this->fecha_fin)){
                $this->addError('fecha_fin', 'La fecha fin no puede estar vacía.');
                return;
            }
            $end_date = $this->fecha_fin;
            if(!$this->hasFormat){
                $start_date = \DateTime::createFromFormat('d/m/Y H:i', $start_date)->format('Y-m-d H:i');
                $end_date = \DateTime::createFromFormat('d/m/Y H:i', $end_date)->format('Y-m-d H:i');
            }
            $start_time = strtotime($start_date);
            $end_time = strtotime($end_date);
            if (!$this->hasErrors('fecha_fin') && $end_time < $start_time) {
                $this->addError('fecha_fin', 'La fecha fin no es válida.');
            }
        }
    }
    
    public function getFechaInicioDB(){
        if(!$this->hasFormat){
            $fechaI = $this->fecha_inicio;
            return \DateTime::createFromFormat('d/m/Y H:i', $fechaI)->format('Y-m-d H:i');
        }
        return $this->fecha_inicio;
    }
    
    public function getFechaFinDB(){
        if(!$this->hasFormat){
            return \DateTime::createFromFormat('d/m/Y H:i', $this->fecha_fin)->format('Y-m-d H:i');
        }
        return $this->fecha_fin;
    }
    
    
    public function getNombreEvento(){
        $usuario = Yii::$app->user->identity;
        $esDoctor = $usuario->tipo == Usuario::DOCTOR;
        return $this->privacidad == 0 || $esDoctor ? $this->nombre : "Evento privado";
    }
    
    
    public function getDescripcionEvento(){
        $usuario = Yii::$app->user->identity;
        $esDoctor = $usuario->tipo == Usuario::DOCTOR;
        return $this->privacidad == 0 || $esDoctor ? $this->descripcion : "Evento privado";
    }
    
    public function getFechaFinEvento(){
        return $this->todo_dia == 1 ? "" : $this->fecha_fin;
    }
    
    public function tododiaGoogleCalendar(){
        $diff = Utils::diferenciaFechas($this->fechaInicioDB, $this->fechaFinDB);
        if ( \intval($diff->format('%d')) == 1 ){
            $this->todo_dia = true;
            $this->fecha_inicio = date('d/m/Y', strtotime($this->fechaInicioDB ) );
        } else {
            $this->todo_dia = false;
        } 
    }
    
    
}
