<?php

namespace app\models;

use Yii;
use app\customs\Utils;


/**
 * This is the model class for table "documento_paciente".
 *
 * @property int $id
 * @property int $id_paciente
 * @property string $documento_src_nombre
 * @property int $tipo
 * @property string $fecha
 * @property string $rutaParaVista
 *
 * @property Paciente $paciente
 */
class DocumentoPaciente extends \yii\db\ActiveRecord
{
    public $documento;
    const TIPO_IMG = 1;
    const TIPO_PDF = 2;
    const TIPO_DOC = 3;
    const TIPO_XLS = 4;
    
    const EXT_PNG = 'png';
    const EXT_JPG = 'jpg';
    const EXT_JPEG = 'jpeg';
    const EXT_GIF = 'gif';
    const EXT_PDF = 'pdf';
    const EXT_DOC = 'doc';
    const EXT_DOCX = 'docx';
    const EXT_XLS = 'xls';
    const EXT_XLSX = 'xlsx';
    const PATH_URL = 'url';
    const PATH_SRC = 'src';
    
    const POST_METHOD = 'post_method';
    const AJAX_METHOD = 'ajax_method';
    
    
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'documento_paciente';
    }
    
    
    public function scenarios()
    {
        $scenarios = parent::scenarios();
        $scenarios[self::POST_METHOD] = ['id_paciente', 'documento_src_nombre', 'descripcion', 'tipo', 'fecha', 'documento'];
        $scenarios[self::AJAX_METHOD] = ['id_paciente', 'documento_src_nombre', 'descripcion', 'tipo', 'fecha'];
        return $scenarios;
    }    

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id_paciente', 'documento_src_nombre', 'tipo', 'fecha'], 'required'],
            [['id_paciente', 'tipo'], 'integer'],
            [['documento'], 'file',
                'extensions' => ['jpg', 'gif', 'png', 'jpeg', 'pdf', 'doc', 'docx', 'xlsx', 'xls'],
                'maxSize' => 600 * 600,
                'maxFiles' => 6,
                'mimeTypes' => ['image/*', 'application/*'],
                'maxSize' => 1024 * 1024 * 3,
                'tooBig' => 'El tamaño máximo permitido es 5MB',
            ],
            [['fecha'], 'safe'],
            [['documento_src_nombre', 'descripcion'], 'string', 'max' => 255],
            [['id_paciente'], 'exist', 'skipOnError' => true, 'targetClass' => Paciente::className(), 'targetAttribute' => ['id_paciente' => 'id']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'id_paciente' => 'Id Paciente',
            'documento_src_nombre' => 'Documento Src Nombre',
            'descripcion' => 'Descripción',
            'documento' => 'Adjuntar Archivos',
            'tipo' => 'Tipo',
            'fecha' => 'Fecha del Documento',
        ];
    }
    
    /**
     * 
     * @param objet $model app\models\Adjunto
     * @param int $citaId
     */
    public static function subirDocumentos($documentos,$idP,$fecha){
        foreach($documentos as $documento ){
            $model = new DocumentoPaciente();
            list ($nombre, $ext) = self::nombreExt($documento->name);
            $src_nombre = Utils::eliminarTildes($nombre).'_'.$idP.Yii::$app->security->generateRandomString(5).".{$ext}";
            $model->id_paciente = $idP;
            $model->fecha = $fecha;
            $model->documento_src_nombre = $src_nombre;
            $model->tipo = self::getTipoDocumento($ext);
            Yii::$app->params['uploadPath'] = Yii::$app->basePath .Yii::$app->params['urlAdjuntoCita'].self::getRutaExtension($ext);
            $path = Yii::$app->params['uploadPath'] . $model->documento_src_nombre;
            $documento->saveAs($path);
            $model->save();
            unset($model);
            
        }
        return true;
    }
    
    
    /**
     * @param $documentoNombre
     * @return $array
     */
    public static function nombreExtDocumento($documentoNombre) {
        $nombreExt = explode('.', $documentoNombre);
        $ext = array_pop($nombreExt); 
        $nombre = strtr(implode('-', $nombreExt), [' ' => '-', '_' => '-']);
        $nombre_clean = Utils::eliminarTildes($nombre);
        return [ $nombre_clean, $ext ];
    }

    /**
     * Gets query for [[Paciente]].
     *
     * @return \yii\db\ActiveQuery
     */
    public function getPaciente()
    {
        return $this->hasOne(Paciente::className(), ['id' => 'id_paciente']);
    }
    
    public function getDocumentoDescripcion()
    {
        return $this->descripcion == null ? 'No definido' : $this->descripcion; 
    }
    
    /**
     * Retorna la ruta del archivo paraser llamado
     * @param string $ext extension del archivo 
     */
    public static function getSrcDocumento($ext)
    {
        return Yii::$app->homeUrl.self::getPathExtension($ext, self::PATH_SRC);
    }
    
    /**
     * Retorna la ruta para subir el archivo
     * @param string $ext extension del archivo 
     */
    public static function getUrlDocumento($ext)
    {
        return Yii::$app->basePath . Yii::$app->params['urlAdjuntoCita'] . self::getPathExtension($ext, self::PATH_URL);
    }
    
    /**
     *
     * @return string
     */
    public function getRutaParaVista(){
        return Yii::$app->homeUrl.self::getRutaTipo($this->tipo).$this->documento_src_nombre;
    }
    
    public static function getRutaTipo($tipo)
    {  
        switch ($tipo){
            case self::TIPO_IMG: 
                return Yii::$app->params['srcAdjuntoImg']; 
            case self::TIPO_PDF: 
                return Yii::$app->params['srcAdjuntoPdf']; 
            case self::TIPO_DOC: 
                return Yii::$app->params['srcAdjuntoDoc']; 
            case self::TIPO_XLS: 
                return Yii::$app->params['srcAdjuntoXls']; 
        }
        
        
    }
    
    
    private static function getPathExtension($ext, $path = self::PATH_URL)
    {
        switch ($ext){
            case self::EXT_PDF: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoPdf' : 'srcAdjuntoPdf']; 
            case self::EXT_DOC: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoDoc' : 'srcAdjuntoDoc'];
            case self::EXT_DOCX: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoDoc' : 'srcAdjuntoDoc'];     
            case self::EXT_XLS: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoXls' : 'srcAdjuntoXls'];
            case self::EXT_XLSX: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoXls' : 'srcAdjuntoXls'];         
            default: 
                return Yii::$app->params[$path == self::PATH_URL ? 'urlAdjuntoImg' : 'srcAdjuntoImg']; 
        }
    }
    
    public static function getTipoDocumento($ext)
    {
        switch ($ext){
            case self::EXT_PDF: 
                return self::TIPO_PDF; 
            case self::EXT_DOC: 
                return self::TIPO_DOC;
            case self::EXT_DOCX: 
                return self::TIPO_DOC;     
            case self::EXT_XLS: 
                return self::TIPO_XLS;
            case self::EXT_XLSX: 
                return self::EXT_XLSX;   
            default: 
                return self::TIPO_IMG; 
        }
    }
    
    public static function minImageCard($tipo,$ad)
    {
        switch ($tipo){
            case self::TIPO_DOC:
                return Yii::$app->homeUrl.'img/word-olivia.jpg';
            case self::TIPO_PDF:
                return Yii::$app->homeUrl.'img/pdf-olivia.jpg';
            case self::TIPO_XLS:
                return Yii::$app->homeUrl.'img/excel-olivia.jpg';
            default :
                return $ad->urlAdjunto;
        }
    }
    
    public static function getMinPreview($path,$ext)
    {
        switch ($ext){
            case self::EXT_DOC:
                return Yii::$app->homeUrl.'img/word-olivia.jpg';
            case self::EXT_DOCX:
                return Yii::$app->homeUrl.'img/word-olivia.jpg';
            case self::EXT_PDF:
                return Yii::$app->homeUrl.'img/pdf-olivia.jpg';
            case self::EXT_XLS:
                return Yii::$app->homeUrl.'img/excel-olivia.jpg';
            case self::EXT_XLSX:
                return Yii::$app->homeUrl.'img/excel-olivia.jpg';
            default :
                return $path;
        }
    }        
    
    
}
