<?php

namespace app\customs;

use app\models\Cita;
use app\models\repositories\CitaRepository;
use DateTime;
use Google\Service\Transcoder\AdBreak;

class CitaPeriodica{

    const NO_REPETIR = 0;
    const DIARIAMENTE = 1;
    const SEMANALMENTE = 2;
    const MENSUALMENTE = 3;
    const PERSONALIZADO = 4;

    const TOTAL_DIARIAS = 7;
    const TOTAL_SEMANALES = 4;
    const TOTAL_MENSUALES = 12;

    const DIA = 'days';
    const SEMANA = 'week';
    const MES = 'month';

    const DOMINGO = 0;
    const LUNES = 1;
    const MARTES = 2;
    const MIERCOLES = 3;
    const JUEVES = 4;
    const VIERNES = 5; 
    const SABADO = 6;

    const FINAL_POR_FECHA = 'fecha';
    const FINAL_POR_REPETICION = 'repeticiones';

    public static $validarDia = false;

    const DIAS = [
        self::DOMINGO => 'Domingo',
        self::LUNES => 'Lunes',
        self::MARTES => 'Martes',
        self::MIERCOLES => 'Miercoles',
        self::JUEVES => 'Jueves',
        self::VIERNES => 'Viernes', 
        self::SABADO => 'Sabado'
    ];

    public static $periodosMenu = [
        'days' => 'Dias',
        'week' => 'Semana',
        'month' => 'Mes'
    ];

    public static function repeticionesMenu(){ 
        return [
            self::NO_REPETIR => 'No se repite',
            self::DIARIAMENTE => 'Diario por una semana',
            self::SEMANALMENTE => 'Semanal en '.self::diaActual($porNumero=false).' por un mes',
            self::MENSUALMENTE => 'Mensual por un año los '.self::diaActual($porNumero=true).' de mes',
            self::PERSONALIZADO => 'Personalizado',
        ];
    }
    /**
     * Retorna el día actual en número o texto
     * @param bool $porNumero
     * @return string 
    */
    private static function diaActual($porNumero=false){
        return $porNumero ? date('d') : self::DIAS[date('w')];
    }
    /**
     * Crear citas periodicas
     *
     * @param app\models\Cita $cita
     * @param array $procsIDs
     * @param array $costoProcs
     * @param int $idUsuario
     * @return void
     */
    public static function crear($cita,$procsIDs,$costoProcs,$idUsuario){
        $repetirCita = intval($cita->repetirCita);
        if( $repetirCita == self::DIARIAMENTE ){
            self::crearPorRepeticiones($cita,self::TOTAL_DIARIAS,$rango=1,self::DIA,$procsIDs,$costoProcs,$idUsuario);
        }else if($repetirCita == self::SEMANALMENTE){
            self::crearPorRepeticiones($cita,self::TOTAL_SEMANALES,$rango=1,self::SEMANA,$procsIDs,$costoProcs,$idUsuario);
        }elseif($repetirCita == self::MENSUALMENTE){
            self::crearPorRepeticiones($cita,Self::TOTAL_MENSUALES,$rango=1,self::MES,$procsIDs,$costoProcs,$idUsuario);
        } else if($repetirCita == self::PERSONALIZADO){
            $periodo = $cita->periodo;
            $diaSemanaRepeticion = $cita->diaSemanaRepeticion;
            $finalizaPor = $cita->finalizarRepeticionPor;
            $repeticiones = intval($cita->repeticiones);
            $rango = intval($cita->rango);
            self::setValidarDia($cita->fechaDB,$diaSemanaRepeticion,$periodo);
            if( $finalizaPor == self::FINAL_POR_REPETICION){
                self::crearPorRepeticiones($cita,$repeticiones,$rango,$periodo,$procsIDs,$costoProcs,$idUsuario);
            }else if($finalizaPor == self::FINAL_POR_FECHA){
                self::crearPorFecha($cita,$rango,$periodo,$procsIDs,$costoProcs,$idUsuario);
            }

        } 
    }
    /**
     * Crear citas por repeticion
     * @param app\models\Cita $cita
     * @param int $repeticiones
     * @param int $rango
     * @param string $periodo
     * @param array $procsIDs
     * @param array $costoProcs
     * @param int $idUsuario
     * @return void
    */
    private static function crearPorRepeticiones($cita,$repeticiones,$rango,$periodo,$procsIDs,$costoProcs,$idUsuario){
        $fechaBase = $cita->fechaDB;
        $repeticiones = $repeticiones;
        CitaRepository::GuardarBitacoraCita($cita->id, $cita->estado, $idUsuario, $cita->cita_periodica_token, "1-$repeticiones");
        for($i=1; $i<$repeticiones; $i++){
            $fecha = !self::$validarDia 
                    ? \date('Y-m-d H:i', strtotime($fechaBase.' +'.($rango*$i).' '.$periodo ))
                    : self::validarFechaPorDiaSemana($cita->fechaDB,$cita->diaSemanaRepeticion,$periodo,$i);
            $model = self::atributosCitaPeriodica($cita,$fecha);
            if($model->save()){
                CitaRepository::GuardarBitacoraCita($model->id, $model->estado, $idUsuario, $model->cita_periodica_token, ($i+1)."-$repeticiones");
                Utils::guardarProcedimientos($model->isNewRecord, $model->id, $procsIDs, $costoProcs, $model->costo);
                unset($model);
            }
        }
    }
    /**
     * Crear citas por fecha
     * @param app\models\Cita $cita
     * @param int $rango
     * @param string $periodo
     * @param array $procsIDs
     * @param array $costoProcs
     * @param int $idUsuario
     * @return void
     */
    private static function crearPorFecha($cita,$rango,$periodo,$procsIDs,$costoProcs,$idUsuario){
        $fechaBase = $cita->fechaDB;
        $hora = explode(' ', $fechaBase)[1];
        $repeticiones = self::obtenerConteoPorFecha($cita);
        $fechaLimite = DateTime::createFromFormat('d/m/Y', $cita->fechaLimite)->format('Y-m-d').' '.$hora;
        CitaRepository::GuardarBitacoraCita($cita->id, $cita->estado, $idUsuario, $cita->cita_periodica_token, "1-$repeticiones");
        $i = 1;
        while(true){
            $fecha = !self::$validarDia 
                    ? \date('Y-m-d H:i', strtotime($fechaBase.' +'.($rango*$i).' '.$periodo ))
                    : self::validarFechaPorDiaSemana($cita->fechaDB,$cita->diaSemanaRepeticion,$periodo,$i);
            if(strtotime($fecha) > strtotime($fechaLimite)) break;
            $model = self::atributosCitaPeriodica($cita,$fecha);
            if($model->save()){
                CitaRepository::GuardarBitacoraCita($model->id, $model->estado, $idUsuario, $model->cita_periodica_token, ($i+1)."-$repeticiones");
                Utils::guardarProcedimientos($model->isNewRecord, $model->id, $procsIDs, $costoProcs, $model->costo);
                unset($model);
            }
            $i++;
        }
    }
    /**
     * Obtener repeticiones por fecha
     *
     * @param app\models\Cita $cita
     * @return int
     */
    private static function obtenerConteoPorFecha($cita){
        $fechaBase = $cita->fechaDB;
        $hora = explode(' ', $fechaBase)[1];
        $periodo = $cita->periodo;
        $diaSemanaRepeticion = $cita->diaSemanaRepeticion;
        $fechaLimite = DateTime::createFromFormat('d/m/Y', $cita->fechaLimite)->format('Y-m-d').' '.$hora;
        $rango = intval($cita->rango);
        $repeticiones = [];
        $i = 0;
        while(true){
            $fecha = !self::$validarDia 
                    ? \date('Y-m-d', strtotime($fechaBase.' +'.($rango*$i).' '.$periodo ))
                    : self::validarFechaPorDiaSemana($fechaBase,$diaSemanaRepeticion,$periodo,$i);
            if(strtotime($fecha) < strtotime($fechaLimite)){
                $repeticiones[] = $fecha;
            } else {
                break;
            }
            $i++;
        }
        return count($repeticiones);
    }
    /**
     * Establecer los atributos de la cita original a la repetitiva
     * @param app\models\Cita $cita
     * @param string $fecha
     * @return app\models\Cita
    */
    private static function atributosCitaPeriodica($cita,$fecha){
        $model = new Cita();
        $model->fecha = $fecha;
        $model->hasFormat = true;
        $model->tipo_cita = $cita->tipo_cita;
        $model->id_suscripcion = $cita->id_suscripcion; 
        $model->id_clinica = $cita->id_clinica;
        $model->estado = $cita->estado;
        $model->tiempo_estimado = $cita->tiempo_estimado;
        $model->costo = $cita->costo;
        $model->asunto = $cita->asunto;
        $model->diagnostico = $cita->diagnostico;
        $model->receta = $cita->receta;
        $model->id_paciente = $cita->id_paciente;
        $model->cita_periodica_token = $cita->cita_periodica_token;
        return $model;
    }
    /**
     * Validar si el dia de la semana es diferente al de la fecha seleccionada
     * @param string $fecha
     * @param int $diaSemana
     * @param string $periodo
     * @return void
     */
    public static function setValidarDia($fecha,$diaSemana,$periodo){
        if($periodo != self::SEMANA){
            self::$validarDia = false;
        } else {
            $diaRepeticion = intval($diaSemana) + 1; 
            $diaPorfecha = intval(date('w', strtotime($fecha))) + 1; 
            self::$validarDia = $diaPorfecha == $diaRepeticion ? false : true;
        }
    }
    /**
     * Validar el dia de la semana selecccionado
     * @param string $fecha
     * @param int $diaSemana
     * @param string $periodo
     * @param int $multiplo multiplo de periodo a agregar a la fecha base
     * @return string
    */
    private static function validarFechaPorDiaSemana($fecha,$diaSemana,$periodo='week',$multiplo){
        $diaSemanaSelec = $diaSemana + 1;
        $diaSemanaporFecha = date('w', strtotime($fecha)) + 1;
        $esDiaSuperior = $diaSemanaSelec > $diaSemanaporFecha;
        $diffDias = $esDiaSuperior ? ($diaSemanaSelec-$diaSemanaporFecha) : ($diaSemanaporFecha-$diaSemanaSelec) ;
        $operacion = $esDiaSuperior ? '+' : '-';
        return date('Y-m-d H:i', strtotime($fecha." +$multiplo $periodo $operacion $diffDias days"));
    }

}