<?php

namespace app\controllers;

use Yii;
use app\models\Cita;
use app\models\web\CitaWeb;
use app\models\Evento;
use app\customs\BaseController;
use yii\helpers\Url;
use yii\filters\VerbFilter;
use yii\web\Response;
use DateTime;
use DateInterval;
use app\models\Usuario;
use app\customs\SesionUtils;
use app\models\RecordatorioDoctor;
use app\models\Seguimiento;
use app\models\Procedimiento;
use app\models\Suscripcion;
use app\customs\Utils;

class CalendarioController extends BaseController {
    //put your code here
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'citas' => ['post'],
                    'eventos' => ['post'],
                ],
            ],
        ];
    }
    
    public function actionIndex(){
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $idSuscripcion = $suscripcion->id;
        $id_clinica = $usuario->tipo == Usuario::DOCTOR 
                ? null 
                : SesionUtils::ClinicaSeleccionada()->id;
        $procedimientos = $suscripcion->tipo == Suscripcion::TIPO_INDIVIDUAL 
                ? Utils::obtenerProcedimientosClinica($usuario->tipo,$id_clinica,$idSuscripcion)
                : Utils::obtenerProcedimientosGrupo();
        return $this->render('index' ,['procedimientos' => $procedimientos ]);
    }
    
    
    private function obtenerProcedimientosClinica($tipoUsuario,$idClinica,$idSuscripcion){
        $qProcedimientos = Procedimiento::find()->alias('p')->where(['id_suscripcion' => $idSuscripcion]);
        if($tipoUsuario == Usuario::ASISTENTE){
            $qProcedimientos->innerJoin('clinica_procedimiento as cp','cp.id_procedimiento = p.id')
            ->andWhere(['cp.id_clinica' => $idClinica]);
        }
        return $qProcedimientos->all();
    }
    
    
    /**
     * Identificadores de cita por color:  
     * agendada(azul): #85c1e9, 
     * completada(verde): #2ecc71, 
     * reprogramada(naranja): #f39c12, 
     * cancelada(rojo): #f1948a,
     * Inasistencia(rojo): #EC7063; 
     * Asistio (): #013220
     */
    public function actionCitas(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $start = $post["start"];
        $end = $post["end"];
        //Filtro para clinicas jerarquizadas e individuales
        $where = ($usuario->tipo == Usuario::DOCTOR && $suscripcion->tipo == Suscripcion::TIPO_INDIVIDUAL)
                ? ['id_suscripcion' => $usuario->suscripcion->id]
                : ['id_suscripcion' => $usuario->suscripcion->id,
                    'id_clinica' => SesionUtils::ClinicaSeleccionada()->id];
        $citas = Cita::find()->where($where)
                ->andWhere([
                    'or',
                    ['between','fecha',$start,$end],
                    ['between','DATE_ADD(fecha,INTERVAL tiempo_estimado MINUTE)',$start,$end] 
                ])
                ->andFilterWhere(['<>','estado',Cita::CANCELADA])->all();
        
     
        $citasJson = [];
        $colorFondo = '';
        $colorBorde = '';
        $now = strtotime(date('Y-m-d H:i'));
        foreach ($citas as $cita) {
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $cita->fecha)->format('Y-m-d H:i');
            if($now > strtotime($fecha) && $cita->estado == Cita::COMPLETADA){
                $colorFondo = '#2ecc71';
                $colorBorde = '#28A45C';
            }else if($now < strtotime($fecha) && ($cita->estado == Cita::CONFIRMADA || $cita->estado == Cita::REPROGRAMADA || $cita->estado == Cita::AGENDADA )){
                $colorFondo = '';
                $colorBorde = '#2232AA';
            } else if ($cita->estado == Cita::INASISTENCIA){
                $colorFondo = '#ec7063';
                $colorBorde = '#B62416';
            }else {
                $colorFondo = '#f39c12';
                $colorBorde = '#C27B0A';
            } 
            $procedimientos = '';
            
            $j = count($cita->procedimientoCitas );
            for($i = 0; $i < $j; $i++){
                if($i+1 < $j){
                    $procedimientos .= $cita->procedimientoCitas[$i]->procedimiento->nombre.', ';  
                } else {
                    $procedimientos .= $cita->procedimientoCitas[$i]->procedimiento->nombre;
                }
            }    

            
            $curr_date=new DateTime($fecha);
            $curr_date->add(new DateInterval('PT' . $cita->tiempo_estimado . 'M'));
            $citasJson[] = [
                    'title' => $cita->paciente->nombreCompleto,
                    'start' => $fecha,
                    'className' => $cita->estado == Cita::CONFIRMADA ? 'fc-event-success fc-event-solid-primary' : 'fc-event-light fc-event-solid-primary',
                    'color' => $colorFondo,
                    'borderColor' => $colorBorde,
                    'type' => 'cita',
                    'esPresencial' => $cita->tipo_cita == Cita::CITA_PRESENCIAL ? true : false,
                    'end' => $curr_date->format('Y-m-d H:i'),
                    'description' => $cita->asunto == '' ? '(No definido)' : $cita->asunto,
                    'procedimientos' => $procedimientos,
                    'editable' => false,
                    'tooltip' => true,
                    'cel' => $cita->paciente->celular == '' ? '(No definido)' : $cita->paciente->celular,
                    'email' => $cita->paciente->email == '' ? '(No definido)' : $cita->paciente->email,
                    'url' => Url::toRoute(['cita/view','id' => $cita->id])
                ];
            $procedimientos = '';
        }
        return $citasJson;
    }
    
        public function actionCitasWeb(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $start = $post["start"];
        $end = $post["end"];
        $where = $usuario->tipo == Usuario::DOCTOR 
                ? ['id_suscripcion' => $usuario->suscripcion->id]
                : ['id_suscripcion' => $usuario->suscripcion->id,
                    'id_clinica' => SesionUtils::ClinicaSeleccionada()->id];
        $citas = CitaWeb::find()->where($where)
                ->andWhere([
                    'or',
                    ['between','fecha',$start,$end],
                    ['between','DATE_ADD(fecha,INTERVAL tiempo_estimado MINUTE)',$start,$end] 
                ])
                ->andFilterWhere(['estado' => CitaWeb::ACTIVA])
                ->all();
        
        $citasJson = [];
        $color = '#A2D9CE';//gris azulado
        $now = strtotime(date('Y-m-d H:i'));
        foreach ($citas as $cita) {
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $cita->fecha)->format('Y-m-d H:i');
            $procedimientos = $cita->primerProcedimiento->nombre;
            $curr_date = new DateTime($fecha);
            $curr_date->add(new DateInterval('PT' . $cita->tiempo_estimado . 'M'));
            $citasJson[] = [
                    'start' => $fecha,
                    'end' => $curr_date->format('Y-m-d H:i'),
                    'title' => $cita->paciente->nombreCompleto,
                    'id' => $cita->id,
                    'className' => 'fc-event-ligth fc-event-solid-primary',
                    'color' => $color,
                    'borderColor' => '#343A65',
                    'type' => 'solicitud',
                    'description' => $cita->comentario == null ? '(No definido)' : $cita->comentario,
                    'procedimientos' => $procedimientos,
                    'editable' => false,
                    'tooltip' => true,
                    'cel' => $cita->paciente->celular == '' ? '(No definido)' : $cita->paciente->celular,
                    'email' => $cita->paciente->email == '' ? '(No definido)' : $cita->paciente->email,
                    'url' => 'javascript:void(0);'
                ];
        }
        return $citasJson;
    }
        
    public function actionEventos(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $id_suscripcion = $suscripcion->id;
        $start = $post["start"];
        $end = $post["end"];
        $esDoctor = $usuario->tipo == Usuario::DOCTOR;
        $Qeventos = Evento::find()->where(['id_suscripcion' => $id_suscripcion])
                ->andWhere([
                'or',
                ['between','fecha_inicio',$start,$end],
                ['between','fecha_fin',$start,$end] 
            ]);
        
        if($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $Qeventos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $eventos = $Qeventos->all();
        $eventosJson = [];
        foreach ($eventos as $evento) {
            $eventosJson[] = [
                    'title' => $evento->nombreEvento,
                    'start' => $evento->fechaInicioDB,
                    'className' => 'fc-event-light fc-event-solid-primary',
                    'color' => '#a569bd',
                    'type' => 'evento',
                    'end' => $evento->fechaFinDB,
                    'allDay' => $evento->todo_dia == null ? false : $evento->todo_dia,
                    'description' => $evento->descripcionEvento,
                    'editable' => false,
                    'tooltip' => false,
                    'url' => Url::toRoute(['evento/view','id' => $evento->id])
                ];
        }
        return $eventosJson;
    }
    
    public function actionRecordatorios(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $idSuscripcion = $suscripcion->id;
        $start = $post["start"];
        $end = $post["end"];
        $recordatoriosD = RecordatorioDoctor::find()
                        ->where(['id_suscripcion' =>$idSuscripcion, 'finalizado'=>0])
                        ->andWhere([
                            'or',
                            ['between','fecha_inicio',$start,$end],
                            ['between','fecha_fin',$start,$end]
                        ]);
        if($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $recordatoriosD->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $recordatorios = $recordatoriosD->all();
        $recordatoriosJson = [];
        $now = date('Y-m-d H:i:s');
        foreach ($recordatorios as $recordatorio){
            $esPaciente = boolval($recordatorio->es_paciente);
            if(strtotime($now) > strtotime($recordatorio->fechaInicioDB) && strtotime($now) > strtotime($recordatorio->fechaFinDB)){
                $color = '#ba4a00';
            } else {
                $color = '#eb984e';
            }
            $recordatoriosJson[] = [
                'title' => $recordatorio->asuntoRecordatorio,
                'start' => $recordatorio->fechaInicioDB,
                'end' => $recordatorio->fechaFinDB,
                'className' => 'fc-event-light fc-event-solid-primary',
                'color'=> $color,
                'type' => 'recordatorio',
                'allDay' => $recordatorio->todo_dia == null ? false : $recordatorio->todo_dia,
                'editable' => false,
                'tooltip' => $usuario->tipo == Usuario::DOCTOR ? true : (($usuario->tipo == Usuario::ASISTENTE && $recordatorio->privacidad ) ? false : true),
                'paciente' => $esPaciente ? $recordatorio->nombrePaciente : 'No definido',
                'cel' => $esPaciente ? ($recordatorio->paciente->celular == '' ? 'No definido' : $recordatorio->paciente->celular) : 'No definido',
                'email' => $esPaciente ? ($recordatorio->paciente->email == '' ? 'No definido' : $recordatorio->paciente->email) : 'No definido', 
                'url' => Url::toRoute(['recordatorio-doctor/view','id' => $recordatorio->id])
            ];
        }        
        return $recordatoriosJson;        
    }
    
    public function actionSeguimientos(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $hoy = date('Y-m-d H:i');
        $usuario = Yii::$app->user->identity;
        $qSeguimientos  = Seguimiento::find()
            ->where(['id_suscripcion' => $usuario->suscripcion->id])
            ->andWhere(['estado'=> Seguimiento::PENDIENTE])
            ->andWhere(['<=','fecha_seguimiento',date('Y-m-d')]);
        if( $usuario->suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA ){
            $qSeguimientos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id ]);
        }
        $seguimientos = $qSeguimientos->all();
        $result = [];
        if(count($seguimientos) > 0){
            $result[] = [
                'title' => 'Seguimientos Pendientes',
                'start' => $hoy,
                'end' => \date('Y-m-d H:i', \strtotime($hoy.'+5 hours')),
                'className' => 'fc-event-light fc-event-solid-info',
                'color'=> '#ff0800',
                'type' => 'seguimientos',
                'allDay' => true,
                'editable' => false,
                'url' => Url::toRoute(['seguimiento/index']) 
            ];
        }
        return $result;
    }
    
    
    public function excludeFromSecurity(){
         return [
             'citas',
             'citas-web',
             'eventos', 
             'recordatorios', 
             'seguimientos'];
    }
    
    public function beforeAction($action)
    {            
        if ($action->id == 'citas' || $action->id == 'citas-web' ||$action->id == 'eventos' || $action->id == 'recordatorios' || $action->id == 'seguimientos') {
            $this->enableCsrfValidation = false;
        }

        return parent::beforeAction($action);
    }
}
