<?php

namespace app\controllers;

use Yii;
use app\models\web\CitaWeb;
use app\models\Cita;
use app\models\Seguimiento;
use app\models\Paciente;
use app\models\ListaEspera;
use app\models\ProcedimientoCita;
use app\customs\BaseController;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\web\Response;
use app\customs\SesionUtils;
use app\models\UrlSuscripcion;
use app\models\repositories\CitaRepository;
use app\models\Suscripcion;

class SolicitudCitaController extends BaseController
{
     /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'denegar' => ['POST'],
                ],
            ],
        ];
    }
    
    public function actionIndex(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $hoy = date('Y-m-d H:i');
        $usuario = Yii::$app->user->identity;
        $condicion = ['id_suscripcion' => $usuario->suscripcion->id, 'estado' => CitaWeb::ACTIVA ];
        $model = CitaWeb::find()->where( $condicion )->all();
        $solicitudes = [];
        foreach ( $model as $sl ){
            $solicitudes[] = [
                'id' => $sl->id,  
                'fecha' => $sl->fechaCompleta,
                'fecha_expirada' => strtotime( $hoy ) > strtotime( $sl->fechaDB )  ? true : false,
                'tipoCita' => $sl->tipo_cita == CitaWeb::CITA_EN_LINEA ? 'Cita en línea' : 'Cita presencial', 
                'procedimientos' => $sl->primerProcedimiento->nombre, 
                'comentario' => ($sl->comentario != null || $sl->comentario != '') ? $sl->comentario : '(Sin comentarios del paciente)',
                'paciente' => [
                    'id' => $sl->paciente->id,
                    'nombreCompleto' => $sl->paciente->nombreCompleto
                ]
            ];
        }
        return ['Correcto' => true, 'Solicitudes'=> $solicitudes ];
    }
    

    public function actionAceptar($id){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $model = $this->findModel($id);
        $model->scenario = CitaWeb::SCENARIO_UPDATE;
        if($model->id_suscripcion != $usuario->suscripcion->id){
            return ['Correcto' => false , 'Mensaje' => 'No se pudo completar la acción' ];
        }
        if( $model->id_cita_olivia != null ){
            return ['Correcto' => false , 'Mensaje' => 'La solicitud ya ha sido aceptada previamente' ];
        }
        $model->estado = CitaWeb::ACEPTADO;
        $cita = $this->solicitud_a_cita($model);
        if( $cita != null ){
            $model->id_cita_olivia = $cita->id;
            if( $model->save() ){
                $this->enviarMensajeProcesoSolicitud($model, 'aceptar_solicitud_cita');
                return [ 'Correcto' => true, 'Mensaje'=> 'Proceso realizado correctamente' ];
            } else {
                return [ 'Correcto' => false, 'Mensaje'=> 'Ocurrio un error' ];
            }
        }
    }
    
    public function actionDenegar(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $post = Yii::$app->request->post();
        $model = $this->findModel($post['id']);
        $model->scenario = CitaWeb::SCENARIO_UPDATE;
        if( $model->id_suscripcion != $usuario->suscripcion->id ){
            return ['Correcto' => false , 'Mensaje' => 'No se pudo completar la acción' ];
        }
        if( $model->id_cita_olivia != null ){
            return ['Correcto' => false , 'Mensaje' => 'La solicitud ya ha sido denegada previamente' ];
        }
        $model->estado = CitaWeb::DENEGADO;
        if( $model->save() ){
            if(\boolval($post['lista_espera'])){
                $this->enviarMensajeProcesoSolicitud($model, 'denegar_solicitud_ls_espera');
                $res = $this->agregarEnListaDeEspera($model);
                return $res->save() ? [ 'Correcto' => true, 'Mensaje'=> 'La solicitud ha sido denegada y el paciente agregado a su lista de espera'] : ['Correcto' => false, 'e'=>$res->getErrors()];
            } else {
                $this->enviarMensajeProcesoSolicitud($model, 'denegar_solicitud_cita');
                return [ 'Correcto' => true, 'Mensaje'=> 'La solicitud ha sido denegada' ];
            }
        } else {
            return ['Correcto' => false , 'Mensaje' => 'No se pudo completar la acción. Ocurrio un error', 'msg' => $model->errors ];
        }
    }
    
    public function actionView($id){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $model = $this->findModel($id);
        if($model->id_suscripcion != $usuario->suscripcion->id){
            return ['Correcto' => false , 'mensaje' => 'No se puede obtner la descripción de este elemento' ];
        }
        return [
            'Correcto' => true,
            'model' => [
                'id' => $model->id,
                'fecha_expirada' => strtotime( $model->fechaDB ) < strtotime( date('Y-m-d H:i:s') )  ? true : false,
                'idSuscripcion' => $model->id_suscripcion, 
                'fecha' => $model->fechaCompleta, 
                'tipoCita' => $model->tipo_cita == CitaWeb::CITA_EN_LINEA ? 'Cita en línea' : 'Cita presencial', 
                'procedimientos' => $model->primerProcedimiento->nombre, 
                'comentario' => $model->comentario != null ? $model->comentario : '(Sin comentarios del paciente)',
                'paciente' => [
                    'id' => $model->paciente->id,
                    'nombreCompleto' => $model->paciente->nombreCompleto,
                    'nombreCorto' => $model->paciente->nombreCorto,
                    'celular' => $model->paciente->celular,
                    'email' => $model->paciente->email,
                    
                ]
            ],
        ];
    }
    
    public function actionSolicitudASeguimiento(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $post = Yii::$app->request->post();
        $seguimiento = new Seguimiento();
        $solicitud = $this->findModel($post['id']);
        $solicitud->scenario = CitaWeb::SCENARIO_UPDATE;
        $hoy = date('Y-m-d');
        if( $solicitud->id_suscripcion != $usuario->suscripcion->id ){
            return ['Correcto' => false , 'mensaje' => 'No se pudo completar la acción' ];
        }
        $solicitud->estado = CitaWeb::DENEGADO;
        $solicitud->save();
        $seguimiento->id_suscripcion = $usuario->suscripcion->id;
        $seguimiento->id_paciente = $solicitud->paciente->id_paciente_olivia != null ? $solicitud->paciente->id_paciente_olivia 
                                    : $this->pacienteWeb_a_Olivia( $solicitud )->id;
        $seguimiento->motivo = $post['motivo'];
        $seguimiento->fecha_incidencia = date('Y-m-d H:i:s');
        $seguimiento->fecha_seguimiento = date('d/m/Y', strtotime($hoy.'+15 days'));
        $seguimiento->interaccion = '¡Llamarle al paciente!';
        $seguimiento->hora_seguimiento = SesionUtils::getHoraEntradaClinica($usuario);
        $seguimiento->estado = Seguimiento::PENDIENTE;
        $seguimiento->origen = 1;
        return $seguimiento->save() ? ['Correcto' => true] : ['Correcto' => false, 'mensaje' => 'Hubo un error no se pudo completar la acción', 'e'=>$seguimiento->getErrors()];
    }
    
    /**
     * Convertir la solicitud de cita a una cita
     * @param app\models\web\CitaWeb $model
     */
    private function solicitud_a_cita($model){
        $cita = new Cita();
        $usuario = Yii::$app->user->identity;
        $idPac = $model->paciente->id_paciente_olivia != null ? $model->paciente->id_paciente_olivia : $this->pacienteWeb_a_Olivia($model)->id;
        $cita->estado = Cita::AGENDADA;
        $cita->id_suscripcion = $model->id_suscripcion;
        $cita->id_clinica = $model->id_clinica;
        $cita->fecha = $model->fecha;
        $cita->id_paciente = $idPac;
        $cita->costo = $model->costo;
        $cita->tiempo_estimado = $model->tiempo_estimado;
        $cita->tipo_cita = $model->tipo_cita;
        $cita->asunto = $model->comentario;
        if( $cita->save() ){
            CitaRepository::GuardarBitacoraCita($cita->id, Cita::AGENDADA, $usuario->id);
            $this->guardarProcedimientos($cita->id, $model->procedimientosCitaWeb);
            return $cita; 
        }
        return null;
        
    }
    
    private function guardarProcedimientos($idCita, $procedimientos){
        foreach ($procedimientos as $k => $prc){
            $procedimientoCita = new ProcedimientoCita();
            $procedimientoCita->id_cita = $idCita;
            $procedimientoCita->id_procedimiento = $prc->procedimiento->id;
            $procedimientoCita->orden = $k;
            $procedimientoCita->costo_editado = $prc->procedimiento->costo;
            $procedimientoCita->save();
        }
    }
    
    /**
     * Enviar Mensaje de proceso de solicitud de cita
     * @param app\models\web\CitaWeb $model
     * @param string $plantilla
     */
    private function enviarMensajeProcesoSolicitud($model, $plantilla){
        $articulo = SesionUtils::armarArticulo($model->id_suscripcion, false, false);
        $linkTeleconsulta = SesionUtils::getLlaveValor('link_teleconsulta')->valor;
        $usuario_url = UrlSuscripcion::find()->where(['id_suscripcion'=>$model->id_suscripcion])->one();
        $link_solicitud = Yii::$app->params['shortUrlServer'].'me/?'.$usuario_url->usuario;
        if( $model->paciente->email != '' ){
            $email = $model->paciente->email;
            Yii::$app->mailer->compose($plantilla,['model' => $model, 'articulo'=>$articulo, 'linkTeleconsulta' => $linkTeleconsulta, 'link_solicitud' => $link_solicitud])
                    ->setFrom(['citas@oliviahealth.com' => 'Olivia Health'])
                    ->setTo($email)
                    ->setSubject( 'Proceso de solicitud de cita con '. $articulo )
                    ->send();
        }
    }
    
    /**
     * Pasar el paciente de solicitud de cita a la lista de espera
     * @param app\models\web\CitaWeb $model
     */
    private function agregarEnListaDeEspera($model){
        $fecha = \date('d/m/Y', strtotime($model->fecha));
        $hora = \date('g:i A', strtotime($model->fecha));
        $idPac = $model->paciente->id_paciente_olivia != null ? $model->paciente->id_paciente_olivia : $this->pacienteWeb_a_Olivia($model)->id;
        $lsEspera = new ListaEspera();
        $lsEspera->id_suscripcion = $model->id_suscripcion;
        $lsEspera->id_paciente = $idPac;
        $lsEspera->fecha_creacion = date('Y-m-d H:i:s');
        $lsEspera->comentario = 'En lista de espera debido a una solicitud de cita denegada para el '.$fecha. ' a las '.$hora;
        $lsEspera->estado = ListaEspera::PENDIENTE;
        return $lsEspera;
    }
    
    /**
     * Realizar el registro del paciente de la solicitud de cita web a un paciente en Olivia
     * @param app\models\web\CitaWeb $model
     * @return app\models\Paciente
     */
    private static function pacienteWeb_a_Olivia($model){
        $usuario = Yii::$app->user->identity;
        $suscripcion = $usuario->suscripcion;
        $pcOlivia = Paciente::find()
                ->where(['id_suscripcion' => $suscripcion->id ])
                ->andWhere([
                    'OR',
                    [ 'email' => $model->paciente->email ],
                    [ 'celular' => $model->paciente->celular ]
                ])->one();
        if( empty( $pcOlivia ) ){
            $pcOlivia = new Paciente();
            $pcOlivia->scenario = Paciente::SCENARIO_WEB_TO_OLIVIA;
            $pcOlivia->id_suscripcion = $model->id_suscripcion;
            $pcOlivia->nombre = $model->paciente->nombre;
            $pcOlivia->apellido = $model->paciente->apellido;
            $pcOlivia->email = empty($model->paciente->email) ?  null : $model->paciente->email;
            $pcOlivia->celular = empty($model->paciente->celular) ?  null : $model->paciente->celular;
            $pcOlivia->fecha_creacion = date('Y-m-d H:i:s');
            $grupos = $suscripcion->tipo == Suscripcion::TIPO_INDIVIDUAL 
                        ? []
                        : SesionUtils::gruposClinicaSeleccionada();
            if( $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA ){
                $pcOlivia->id_clinica = SesionUtils::ClinicaSeleccionada()->id; 
                $pcOlivia->estado = Paciente::ESTADO_ACTIVO; 
                $pcOlivia->id_grupo = count($grupos) > 0 ? $grupos[0]->id : null;
            }
            $pcOlivia->save();
        }
        return $pcOlivia; 
    }
    
    /**
     * Finds the CitaWeb model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Clinica the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = CitaWeb::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }
    
    public function excludeFromSecurity() {
        return [
            'index',
            'view',
            'aceptar',
            'denegar',
            'solicitud-a-seguimiento'
            ];
    }
    
    public function beforeAction($action) {
        if( $action->id == 'denegar' || $action->id == 'aceptar' ){
            $this->enableCsrfValidation = false;
        }
        return parent::beforeAction($action);
    }
    
}
