<?php

namespace app\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\web\Response;
use yii\filters\VerbFilter;
use app\models\LoginForm;
use app\customs\BaseController;
use app\models\Usuario;
use app\models\Cita;
use yii\helpers\ArrayHelper;
use app\models\BitacoraCita;
use app\customs\SesionUtils;
use yii\helpers\Url;
use yii\db\Expression;
use app\models\Feedback;
use app\models\BitacoraMensajeLibre;
use app\models\repositories\SecurityFunctions;
use app\models\BitacoraCitaMensaje;
use app\models\Evento;
use app\models\RecordatorioDoctor;
use app\models\Paciente;
use Stripe\Stripe;
use app\customs\StripeUtils;
use app\models\FormaPago;
use yii\helpers\Html;
use app\models\Seguimiento;
use ClickSend\Configuration;
use ClickSend\Api\SMSApi;
use GuzzleHttp\Client;
use ClickSend\Model\SmsMessage;
use ClickSend\Model\SmsMessageCollection;
use app\models\web\CitaWeb;
use app\models\Suscripcion;

class SiteController extends BaseController
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'only' => ['logout'],
                'rules' => [
                    [
                        'actions' => ['logout'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],
        ];
    }

    /**
     * Displays homepage.
     *
     * @return string
     */
    public function actionIndex()
    {
        $this->bodyClass ="actividad-body";
        return $this->render('index');
    }

    /**
     * Login action.
     *
     * @return Response|string
     */
    public function actionLogin()
    {
        if (!\Yii::$app->user->getIsGuest()) {
            return $this->goHome();
        }
        $model = new LoginForm();
        $cargado=$model->load(Yii::$app->request->post());
        
        if($cargado){
            $usuario = Usuario::findOne(['nombre_usuario' => $model->username]);
            if($usuario != null){
                if($usuario->activo == 1 && $usuario->rol->activo == 1){
                    $usuarioValido=$usuario->validatePassword($model->password);
                    #$usuarioValido=true;
                    if ($usuarioValido && $usuario->activo) {
                        if($usuario->tipo == Usuario::ADMIN)
                        {
                            \Yii::$app->user->setIdentity($usuario);
                            $usuario->login();
                            return $this->goBack();
                        }
                        else
                        {
                            if($usuario->suscripcionActiva)
                            {
                                \Yii::$app->user->setIdentity($usuario);
                                $usuario->login();
                                $user = \Yii::$app->user->identity;
                                
                                if( boolval( $user->suscripcion->stripe_activo ) ){
                                    $dias_prueba = $this->diasPrueba($user);
                                    if($dias_prueba !== null)
                                    {
                                        if($dias_prueba >= 1 && $dias_prueba < 5)
                                        {
                                            \Yii::$app->session->setFlash('warning', '<div style="font-size:1.3rem;" class="mt-2">Quedan <strong>'.$dias_prueba.'</strong> días para termine tu período de prueba. ¡Añade tu forma de pago haciendo <strong>'.Html::tag('a','click aqui',['href'=>Url::toRoute(['forma-pago/index']),'style'=>'color:#fff;text-decoration:none;']).'</strong> para seguir ofreciéndole la mejor atención a tus pacientes!</div>');
                                        } 
                                        else if($dias_prueba == 1 || $dias_prueba == 0)
                                        {
                                            \Yii::$app->session->setFlash('danger', '<div style="font-size:1.3rem;" class="mt-2">¡Tu período de prueba ha terminado! ¡Haz <strong>'.Html::tag('a','click aqui',['href'=>Url::toRoute(['forma-pago/index']),'style'=>'color:#fff;text-decoration:none;']).'</strong> y añade una forma de pago para que no vuelvas a perder citas!</div>');
                                        }
                                    }
                                    
                                }
                                return $this->goBack();

                            }
                            else
                            {
                                $model->addError('hiddenBox', 'Suscripción no activa. Verifique estar al tanto de su pago');
                            }
                        }
                    }
                    else{
                        $model->addError('hiddenBox', 'El usuario o contraseña son incorrectos.');
                    }
                }
                else{
                    $model->addError('hiddenBox', 'Lo sentimos, no tiene acceso al sistema, favor comuniquese con el administrador.');
                }
            }
            else{
                $model->addError('hiddenBox', 'El usuario no existe en el sistema.');
            }
        }
        $this->layout = false;
        return $this->render('login', [
            'model' => $model,
        ]);
    }
    
    /**
     * 
     * @param app\models\Usuario $usuario
     * @return int
     */
    private function diasPrueba($usuario){
        $dias = null;
        $secretStripeKey = \Yii::$app->params['strSKey'];
        $publicStripeKey = \Yii::$app->params['strPKey'];
        Stripe::setApiKey($secretStripeKey);
        $subscription = null;
        $trial_end = null;
        $fecha = date('Y-m-d');
        $id = $usuario->tipo == Usuario::DOCTOR ? $usuario->suscripcion->usuario->id : $usuario->usuarioSuscripcion->suscripcion->usuario->id; 
        if($usuario->tipo == Usuario::DOCTOR){
            $formaPago = FormaPago::findOne(['id'=>$id]);
            if($formaPago != null){
                $customerSource = null;
                try{
                    $subscription = StripeUtils::ObtenerSuscripcion($formaPago->subcription_id);
                    $_trial = $subscription->trial_end;
                    $trial_f_end = date('Y-m-d',$_trial);
                    if(strtotime($fecha) < strtotime($trial_f_end)){
                        $fecha_hoy = date_create($fecha);
                        $trial_end = date_create($trial_f_end);
                        $diff = date_diff($fecha_hoy, $trial_end);
                        $dias = $diff->format('%d');
                    } 
                    if(strtotime($fecha) == strtotime($trial_f_end)){
                       $dias = 0; 
                    }
                } catch (\Exception $ex){
                     \Yii::$app->session->setFlash("warning","Existe un error con su forma de pago, por favor revise su cuenta.");
                }
            }
        } 
        return $dias;
    }
    
    /**
     * Logout action.
     *
     * @return Response
     */
    public function actionLogout()
    {
        SesionUtils::LogOut();
        Yii::$app->user->logout();
        return $this->goHome();
    }

    
    
    public function actionDesautorizado()
    {
        return $this->render('desautorizado');
    }
    
    public function actionForbidden()
    {
        return $this->render('forbidden');
    }
        
    public function actionNoSuscripcion()
    {
        return $this->render('no-suscripcion');
    }
    
    public function actionNotificacionesClinica(){
        Yii::$app->response->format = Response::FORMAT_JSON;
        $securityFunctions = new SecurityFunctions;
        $meses = ['Enero','Febrero','Marzo','Abril','Mayo','Junio','Julio','Agosto','Septiembre','Octubre','Noviembre','Diciembre'];
        $post = Yii::$app->request->post();
        $usuario = Yii::$app->user->identity;
        $now = date("Y-m-d");
        $date = $now;
        $fechaPanel = date('d')." de ".$meses[date('n')-1]. " del ".date('Y') ;
        if(isset($post['fecha']) && $post['fecha'] != null ){
            $date = \DateTime::createFromFormat('d/m/Y', $post['fecha'])->format('Y-m-d');
            $f = explode('-', $date);
            $fechaPanel = $f[2]." de ".$meses[$f[1]-1]. " del ".date('Y');
        }   
        $sortActivity = filter_input(INPUT_POST, "sort");
        $notificaciones = ArrayHelper::merge($this->obtenerCitas($usuario,$date,$securityFunctions), $this->obtenerBitacoraCitas($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerFeedbacks($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerBitacoraMensaje($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerBitacoraRecordatorios($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerEventosDia($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerRecordatoriosDoctorDia($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerPacientes($usuario,$date));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerSeguimientos($usuario));
        $notificaciones = ArrayHelper::merge($notificaciones, $this->obtenerSolicitudesCita($usuario,$date));
        if($sortActivity == 'SORT_ASC'){ 
            ArrayHelper::multisort($notificaciones, 'date',SORT_ASC); 
        } else if ($sortActivity == 'SORT_DESC'){ 
            ArrayHelper::multisort($notificaciones, 'date',SORT_DESC);              
        } else {
            $date = date("Y-m-d H:i");
            $intervalSubs = strtotime("-30 minutes", strtotime($date));
            $dateSubs = date("Y-m-d H:i", $intervalSubs);
            ArrayHelper::multisort($notificaciones, 'date',SORT_ASC);
            $recentNotif = [];
            foreach ($notificaciones as $notificacion) {
                if($notificacion['date'] > strtotime($dateSubs)){
                    $recentNotif[] = $notificacion;
                }
            }
            $notificaciones = $recentNotif;

        }
        $toolCita = $securityFunctions->accionPermitida("confirmar-cita", "cita") ? true : false;
        return ['Correcto'=>true,'Notificaciones'=>$notificaciones,'TipoUsuario' => $usuario->tipo, 'toolCita'=>$toolCita, 'fechaPanel'=>$fechaPanel, 'date'=>$date];
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date
     * @param app\models\repositories\SecurityFunctions $securityFunctions
     * @return Array
     */
    private function obtenerCitas($usuario,$date,$securityFunctions){
        $suscripcion = $usuario->suscripcion;
        $qCitas = Cita::find()->where(['id_suscripcion' => $suscripcion->id])
            ->andWhere(['date(fecha)' => new Expression('date(\'' . $date . '\')')])
            ->andWhere(['or',['estado' => Cita::AGENDADA],['estado' => Cita::CONFIRMADA],['estado' => Cita::REPROGRAMADA]]);
        if($usuario->tipo == Usuario::ASISTENTE || ($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA)){
            $qCitas->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $citas = $qCitas->all();
        $result = [];
        $tool = '';
        $titulo = '';
        $now = strtotime(date('Y-m-d H:i'));
        $toolCita = $securityFunctions->accionPermitida("confirmar-cita", "cita") ? true : false;
        foreach($citas as $cita){
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $cita->fecha)->format('Y-m-d H:i');
            if($now > strtotime($fecha)){
                $tool = 'Si';
                $titulo = 'Cita con ' . $cita->paciente->nombreCorto . ((($usuario->tipo == Usuario::DOCTOR || ($usuario->tipo == Usuario::ASISTENTE && $toolCita) )  && $cita->estado < 3) ? ' ¿Asistió el paciente a su cita? ' : '');
            }else {
                $tool = 'No';
                $titulo = 'Cita con ' . $cita->paciente->nombreCorto ; 
            }
            /* @var $toolCita type */
            $result[] = [
                    'fecha' => $cita->fecha,
                    'date' => strtotime($fecha),
                    'titulo' => $titulo,
                    'tipo' => 'cita',
                    'estado' => $cita->estado,
                    'tiempo' => date('g:iA',strtotime($fecha)),
                    'id' => $cita->id,
                    'url' => Url::to(['paciente/view','id'=>$cita->paciente->id, 'idC'=>$cita->id]),
                    'tool' => $tool,
                    'toolCita' => $toolCita
                ];
        }
        return $result;
    }
    /**
     * 
     * @param Usuario $usuario
     * @param string $date
     * @return Array
     */
    private function obtenerBitacoraCitas($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $qBCitas = BitacoraCita::find()->alias('bc')
                ->innerJoin('cita c','c.id = bc.id_cita')
                ->where(['c.id_suscripcion' => $suscripcion->id])
                ->andWhere(['date(bc.fecha)' => new Expression('date(\'' . $date . '\')')]);
        if($usuario->tipo == Usuario::ASISTENTE || $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qBCitas->andWhere(['c.id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $bCitas = $qBCitas->all();
        $result = [];
        foreach($bCitas as $bcita){
            $cita = Cita::findOne(['id' => $bcita->id_cita]);
            $texto = '';
            $nombrePaciente = $cita->paciente->nombreCorto;
            switch ($bcita->estado){
                case Cita::AGENDADA:
                    $texto = 'Cita agendada con ' . $nombrePaciente . ' para el día ' . \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB));
                    break;
                case Cita::CONFIRMADA:
                    $texto = $nombrePaciente . ' ha confirmado su cita para el día ' . \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB));
                    break;
                case Cita::REPROGRAMADA:
                    $texto = 'Se ha reprogramado la cita de ' . $nombrePaciente . ' para el día ' . \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB));
                    break;
                case Cita::COMPLETADA:
                    $texto = 'El paciente ' . $nombrePaciente . ' sí asitió a su cita del  '. \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB));
                    break;
                case Cita::CANCELADA:
                    $texto = 'La cita con ' . $nombrePaciente . ' para el día ' . \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB)) . ', ha sido cancelada';
                    break;
                case Cita::INASISTENCIA:
                    $texto = 'El paciente ' . $nombrePaciente . ' no asitió a su cita del ' . \date('d/m/Y', strtotime($cita->fechaDB)) . ' a las ' . \date('g:iA', strtotime($cita->fechaDB));
                    break;
            }
             
            $result[] = [
                    'fecha' => $bcita->fecha,
                    'date' => strtotime($bcita->fecha),
                    'titulo' => $texto,
                    'tipo' => ($bcita->estado == Cita::COMPLETADA || $bcita->estado == Cita::INASISTENCIA) ? 'cita' : 'paciente',
                    'estado' => $bcita->estado,
                    'tiempo' => date('g:iA',strtotime($bcita->fecha)),
                    'id' => $cita->id,
                    'url' => Url::to(['cita/view','id'=>$cita->id])
                ];
        }
        return $result;
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date
     * @return Array
     */
    private function obtenerFeedbacks($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $qFeedback = Feedback::find()->alias('f')->innerJoin('cita c','c.id = f.id_cita')->where(['c.id_suscripcion' => $suscripcion->id])
            ->andWhere(['date(f.fecha)' => new Expression('date(\'' . $date . '\')')]);
        if($usuario->tipo == Usuario::ASISTENTE || $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qFeedback->andWhere(['c.id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $feedbacks = $qFeedback->all();
        $result = [];
        foreach($feedbacks as $feedback){
            $cita = $feedback->cita;
            $nombrePaciente = $cita->paciente->nombreCorto;
            $texto = $feedback->puntuacion == 1 
                        ? (boolval($feedback->anonimo) ? 'Un paciente ha dejado una retroalimentación negativa sobre su última cita' 
                            : 'El paciente ' . $nombrePaciente . ' ha dejado una retroalimentación negativa sobre su última cita')
                        : ($feedback->puntuacion == 2 
                                ? (boolval($feedback->anonimo) ? 'Has obtenido un puntaje promedio por parte de un paciente' 
                                    :  'Has obtenido un puntaje promedio por parte de ' . $nombrePaciente)
                                : (boolval($feedback->anonimo) ? '¡Felicidades! Has obtenido una retroalimentación positiva por parte de un paciente' 
                                    : '¡Felicidades! Has obtenido una retroalimentación positiva por parte de ' . $nombrePaciente) ) ;
            $result[] = [
                    'fecha' => $feedback->fecha,
                    'date' => strtotime($feedback->fecha),
                    'titulo' => $texto,
                    'tipo' => 'feedback',
                    'estado' => $feedback->puntuacion,
                    'tiempo' => date('g:iA',strtotime($feedback->fecha)),
                    'id' => $cita->id,
                    'url' => 'javascript:void(0);'
                ];
        }
        return $result;
    }
    /**
     * 
     * @param app\models\Usuario $usuario
     * @param string $date
     * @return Array
     */
    private function obtenerBitacoraMensaje($usuario,$date){
        $bitacoras = BitacoraMensajeLibre::find()->where(['id_suscripcion' => $usuario->suscripcion->id])
            ->andWhere(['date(fecha)' => new Expression('date(\'' . $date . '\')')])->all();
        $result = [];
        foreach($bitacoras as $bitacora){
            $result[] = [
                    'fecha' => $bitacora->fecha,
                    'date' => strtotime($bitacora->fecha),
                    'titulo' => 'Se ha enviado un mensaje libre ',
                    'tipo' => 'mensaje',
                    'estado' => 0,
                    'tiempo' => date('g:iA',strtotime($bitacora->fecha)),
                    'id' => $bitacora->id,
                    'url' => 'javascript:void(0);'
                ];
        }
        return $result;
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date  
     * @return Array
     */
    private function obtenerBitacoraRecordatorios($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $comunicacionPaciente = SesionUtils::getLlaveValor('comunicacion_paciente');
        $valComunicacion = \explode("|", $comunicacionPaciente->valor);
        $bmCita = BitacoraCitaMensaje::find()->alias('bmc')
            ->innerJoin('cita c', 'c.id = bmc.id_cita')
            ->where(['c.id_suscripcion' => $suscripcion->id])
            ->andWhere(['date(bmc.fecha)' => new Expression('date(\''.$date. '\')')]);   
        if($usuario->tipo == Usuario::ASISTENTE || $suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $bmCita->andWhere(['c.id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $bmsCitas = $bmCita->all();
        $result = [];
        $titulo = '';
        foreach ($bmsCitas as $bmsCita){
            $cita = Cita::findOne(['id' => $bmsCita->id_cita]);
            $titulo = $this->tituloBitacoraRecordatorios($cita,$bmsCita,$valComunicacion);
            $result[] = [
                'fecha' =>$bmsCita->fecha,
                'date' => strtotime($bmsCita->fecha),
                'titulo' => $titulo,
                'tipo'=> 'recordatorio',
                'tiempo'=> date('g:iA', strtotime($bmsCita->fecha)),
                'id' =>$bmsCita->id,
                'url'=> Url::to(['cita/view', 'id'=>$bmsCita->id_cita])
            ];
        }
        return $result;
    }
    
    /**
     * Armar el título de las notificaciones dependiendo del tipo de envio y los valores de configuración
     * @param objet $cita app\models\Cita
     * @param objet $bmsCita
     * @param array $valComunicacion
     * @return string
     */
    private function tituloBitacoraRecordatorios($cita,$bmsCita,$valComunicacion){
        $titulo = '';
        $paciente = $cita->paciente;
        $smsKey = \in_array('sms', $valComunicacion);
        $emailKey = \in_array('correo', $valComunicacion);
        $esSms = \boolval($bmsCita->es_sms);
        $esEmail = \boolval($bmsCita->es_correo);
        $fechaCita = \date('d/m/Y', \strtotime($cita->fechaDB));
        $horaCita = \date('g:iA', \strtotime($cita->fechaDB));
        $confTxt = "<br><small>(revisar configuración general dando click en \"Configuraciones Generales\" y/o los datos del paciente)</small>";
        $tipoNotif = $bmsCita->estado_cita != Cita::COMPLETADA ? 'un recordatorio' : 'una retroalimentación';
        if(($smsKey && $esSms && $paciente->celular != '') && ($emailKey && $esEmail && $paciente->email != '' )){
            $titulo = 'Se envió '.$tipoNotif.' por sms y correo, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita ;   
        }else if($smsKey && $esSms && $paciente->celular != ''){
            $titulo = 'Se envió '.$tipoNotif.' por sms, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita ;   
        } else if($emailKey && $esEmail && $paciente->email != '') {
            $titulo = 'Se envió '.$tipoNotif.' por correo, a '. $paciente->nombreCorto. ' por su cita del '. $fechaCita . ' a las '. $horaCita ;   
        } else if (($emailKey && $smsKey) && ($esEmail && $esSms ) && $paciente->email != ''){
            $titulo = 'Se envió '.$tipoNotif.' por sms, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita ;   
        } else if((!$emailKey && !$smsKey) && ($esEmail && $esSms) && ($paciente->email != '' && $paciente->celular != '')){
            $titulo = 'Se debió enviar '.$tipoNotif.' por sms y correo, a '. $paciente->nombreCorto. ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        } else if((!$emailKey && !$smsKey) && ($esEmail && $esSms) && ($paciente->email == '' && $paciente->celular == '')){
            $titulo = 'Se debió enviar '.$tipoNotif.' por sms y correo, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        } else if($esEmail && !$emailKey && $cita->paciente->email == ''){
            $titulo = 'Se debió enviar '.$tipoNotif.' por correo, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        } else if($esEmail && !$emailKey && $cita->paciente->email != ''){
            $titulo = 'Se debió enviar '.$tipoNotif.' por correo, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        } else if($esSms && !$smsKey && $cita->paciente->celular == ''){
            $titulo = 'Se debió enviar '.$tipoNotif.' por sms, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        }else {
            $titulo = 'Se debió enviar '.$tipoNotif.' por sms, a '. $paciente->nombreCorto . ' por su cita del '. $fechaCita . ' a las '. $horaCita . $confTxt;   
        }
        return $titulo;
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date  
     * @return Array
     */
    private function obtenerEventosDia($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $qEventos = Evento::find()->where(['id_suscripcion' => $suscripcion->id])
            ->andWhere(['date(fecha_inicio)' => new Expression('date(\'' . $date . '\')')]);
        if($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qEventos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $eventos = $qEventos->all();
        $titulo = '';
        $result = [];
        foreach ($eventos as $evento){
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $evento->fecha_inicio)->format('Y-m-d H:i');            
            $fechaF = \DateTime::createFromFormat('d/m/Y H:i', $evento->fecha_fin)->format('Y-m-d H:i');
            $titulo = ( ($usuario->tipo == Usuario::ASISTENTE && $evento->privacidad ) ? 'Evento privado' :  'Evento: '.$evento->nombre);
            $result[] = [
                'fecha'  => $evento->fecha_inicio,
                'date'   => strtotime($fecha),
                'titulo' => $titulo,
                'tipo'   => 'evento',
                'tiempo' => date('g:iA',strtotime($fecha)).'-'.date('g:iA',strtotime($fechaF)) ,
                'id'     => $evento->id,
                'url'    => Url::to(['evento/view', 'id'=>$evento->id])
            ];
            
        }
        
        return $result;
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date  
     * @return Array
     */
    private function obtenerRecordatoriosDoctorDia($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $qRecordatorios = RecordatorioDoctor::find()->where(['id_suscripcion' => $suscripcion->id])
            ->andWhere(['finalizado'=>0])    
            ->andWhere(['date(fecha_inicio)' => new Expression('date(\'' . $date . '\')')]);
        if($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qRecordatorios->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $recordatorios = $qRecordatorios->all();
        $titulo = '';
        $result = [];
        foreach ($recordatorios as $recordatorio){
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $recordatorio->fecha_inicio)->format('Y-m-d H:i'); 
            $fechaF = \DateTime::createFromFormat('d/m/Y H:i', $recordatorio->fecha_fin)->format('Y-m-d H:i'); 
            $titulo = ( ($usuario->tipo == Usuario::ASISTENTE && $recordatorio->privacidad ) ? 'Recordatorio privado' :  'Recordatorio: '.$recordatorio->asunto);
            $result[] = [
                'fecha'  => $recordatorio->fecha_inicio,
                'date'   => strtotime($fecha),
                'titulo' => $titulo,
                'tipo'   => 'recordatorio-doctor',
                'tiempo' => $recordatorio->todo_dia ? date('g:iA',strtotime($fecha)).'-'.date('g:iA',strtotime($fechaF)) : date('g:iA',strtotime($fecha)),
                'id'     => $recordatorio->id,
                'url'    => Url::to(['recordatorio-doctor/view', 'id'=>$recordatorio->id])
            ];
            
        }
        
        return $result;
    }
    
    /**
     * 
     * @param Usuario $usuario
     * @param string $date  
     * @return Array
     */
    private function obtenerPacientes($usuario,$date){
        $suscripcion = $usuario->suscripcion;
        $qPacientes = Paciente::find()->where(['id_suscripcion' => $suscripcion->id])
            ->andWhere(['date(fecha_creacion)' => new Expression('date(\'' . $date . '\')')]);
        if($suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA){
            $qPacientes->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $pacientes = $qPacientes->all();
        $titulo = '';
        $result = [];
        foreach ($pacientes as $paciente){
            $fecha = $paciente->fecha_creacion;
            $titulo = 'Nuevo paciente: '.$paciente->nombreCompleto.' se ha añadido al sistema';
            $result[] = [
                'fecha'  => $paciente->fecha_creacion,
                'date'   => strtotime($fecha),
                'titulo' => $titulo,
                'tipo'   => 'paciente-nuevo',
                'tiempo' => date('g:iA',strtotime($fecha)),
                'id'     => $paciente->id,
                'url'    => Url::to(['paciente/view', 'id'=>$paciente->id])
            ];
        }
        return $result;
    }
    
    private function obtenerSeguimientos($usuario){
        $hoy = date('Y-m-d H:i');
        $qSeguimientos  = Seguimiento::find()
            ->where(['id_suscripcion' => $usuario->suscripcion->id])
            ->andWhere(['estado'=> Seguimiento::PENDIENTE])
            ->andWhere(['<=','fecha_seguimiento',date('Y-m-d')]);
        if( $usuario->suscripcion->tipo == Suscripcion::TIPO_JERARQUIZADA ){
                $qSeguimientos->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id ]);
        }
        $seguimientos = $qSeguimientos->all();
        $result = [];
        if(count($seguimientos) > 0){
            $result[] = [
                'titulo'    => 'Tienes Seguimientos Pendientes',
                'fecha'     => $hoy,
                'date'      => strtotime($hoy),
                'tipo'      => 'seguimientos',
                'tiempo'    => date('g:iA',strtotime($hoy)),
                'id'        => null,
                'url'       => Url::toRoute(['seguimiento/index']) 
            ];
        }
        return $result;
    }
    
    private function obtenerSolicitudesCita($usuario,$date){
        $qSolicitudes = CitaWeb::find()->where(['id_suscripcion' => $usuario->suscripcion->id])
            ->andWhere(['date(fecha)' => new Expression('date(\'' . $date . '\')')])
            ->andWhere(['estado' => CitaWeb::ACTIVA]);
        if($usuario->tipo == Usuario::ASISTENTE){
            $qSolicitudes->andWhere(['id_clinica' => SesionUtils::ClinicaSeleccionada()->id]);
        }
        $solicitudes = $qSolicitudes->all();
        $result = [];
        foreach( $solicitudes as $sl){
            $fecha = \DateTime::createFromFormat('d/m/Y H:i', $sl->fecha)->format('Y-m-d H:i'); 
            $result[] = [
                    'fecha' => $sl->fecha,
                    'date' => strtotime( $fecha ),
                    'titulo' => 'Solicitud de cita con ' . $sl->paciente->nombreCorto,
                    'tipo' => 'solicitud',
                    'estado' => $sl->estado,
                    'tiempo' => date('g:iA',strtotime($fecha)),
                    'id' => $sl->id,
                    'url' => 'javascript:void(0);'
                ];
        }
        return $result;
    }
    
    public function actionCambiarClinica(){
        
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        
        $idClinica = $post['idc'];
        $clinica = \app\models\Clinica::findOne(['id' => $idClinica]);
        SesionUtils::SeleccionarClinica($clinica);
        Yii::$app->session->set("estado_suscripcion",null);
        Yii::$app->session->set("stripe_activo",null);
        return ['Correcto' => true, 'Doctor' => $clinica->suscripcion->usuario->perfil->nombreCorto,'Clinica' => $clinica->nombre];
    }
    
    public function actionMensajeMuestra(){
        $date = date('H:i:s');
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $username = \yii::$app->params["clickSendU"];
        $password = \yii::$app->params["clickSendP"];
        $csConfig = Configuration::getDefaultConfiguration()
                ->setUsername($username)
                ->setPassword($password);
        $csApi = new SMSApi(new Client(['verify' => false]),$csConfig);
        $smsCollector = new SmsMessageCollection();
        $fechaCita = date('d/m/Y');
        $horaCita = date('H:i', strtotime($date.'+ 30 minutes'));
        $mensaje = 'Hola, recuerda que confirmaste una cita para el '
                .$fechaCita. ' a las '.$horaCita
                .' con tu Doc '.$post['apellido_doctor'];
        $sms = new SmsMessage();
        $sms->setBody($mensaje);
        $sms->setTo($post['celular']);
        $sms->setSource("sdk");
        $smsCollector->setMessages([$sms]);
        $result = $csApi->smsSendPost($smsCollector);
        return ['Correcto'=>true];
    }
    
    
    public function excludeFromSession(){
        return ['login','captcha','error'];
    }
    public function excludeFromSecurity(){
         return ['login','logout', 'index','desautorizado','error','captcha','no-suscripcion','notificaciones-clinica','cambiar-clinica', 'mensaje-muestra'];
    }
    
    public function beforeAction($action)
    {            
        if ($action->id == 'notificaciones-clinica' || $action->id == 'mensaje-muestra') {
            $this->enableCsrfValidation = false;
        }

        return parent::beforeAction($action);
    }
}
