<?php


namespace app\controllers;

use Yii;
use app\models\CobroPagadito;
use app\models\Cita;
use app\models\ProcedimientoCita;
use app\customs\BaseController;
use yii\web\NotFoundHttpException;
use app\customs\SesionUtils;
use ClickSend\Configuration;
use ClickSend\Api\SMSApi;
use GuzzleHttp\Client;
use ClickSend\Model\SmsMessage;
use ClickSend\Model\SmsMessageCollection;
use app\models\repositories\TokenRepository;
use yii\web\Response;
use app\models\BitacoraCitaMensaje;

class CobroPagaditoController extends BaseController{
    
    const Code_N_Fact = '0123456789';
    
    public function actionCobrarCita()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $usuario = Yii::$app->user->identity;
        $shortUrlServer = \Yii::$app->params['shortUrlServer'];
        $post = Yii::$app->request->post();
        $costo = $post['costo'];
        $cita = Cita::find()->where(['id'=>$post['id_cita']])->one();
        if($cita->id_suscripcion != $usuario->suscripcion->id){
            return ['Correcto'=>false, 'mensaje'=>'La cita no pertenece a su suscripción'];
        }
        if($cita->cobroPagadito != NULL){
            return ['Correcto'=>false, 'mensaje'=>'La cita posee un cobro emitido'];
        }
        if($cita->costo != $costo){
            $cita->costo = $costo;
            $cita->save();
            $this->costoProcedimiento($cita);
        }
        $cargos = $this->CALCULAR_TOTAL($costo,$post['metodo_iva'],$post['cargo_pasarela']); 
        $model = new CobroPagadito();
        $model->id_cita = $cita->id;
        $model->n_factura = $this->NUM_FACTURA(CobroPagadito::LONG_NUM_FACTURA);
        $model->total_cita = $costo;
        $model->cargo_pasarela = $cargos['cargo_pasarela'];
        $model->cargar_iva = $post['metodo_iva'];
        $model->iva = $cargos['iva'];
        $model->total_cobrado = $cargos['total_cobrado'];
        $model->metodo_cobro = $post['cargo_pasarela'];
        $model->notas = 'Cita del '.\date('d/m/Y', strtotime($cita->fechaDB)).' a las '.\date('g:iA', strtotime($cita->fechaDB));
        $model->fecha_cobro = date('Y-m-d H:i:s');
        $model->token = TokenRepository::getToken($length = 10);
        if($model->save()){
           $URL_COBRO =  $shortUrlServer.'c/i?t='.$model->token;
           $this->enviar_url_cobro_paciente($cita, \boolval($post['enviar_sms']), \boolval($post['enviar_correo']), $URL_COBRO); 
           return  ['Correcto'=>true, 'URL_COBRO'=>$URL_COBRO]; 
        } else {
            echo "MODEL NOT SAVED";
            print_r($model->getAttributes());
            print_r($model->getErrors());
            exit;
        }
        
    }
    
    /**
     * Envio de url de cobro
     * @param  app\models\Cita $cita
     * @param  boolean $enviar_sms 
     * @param  boolean $enviar_correo 
     * @param  string $URL_COBRO 
     */
    private function enviar_url_cobro_paciente($cita,$enviar_sms,$enviar_correo,$URL_COBRO){
        if ($enviar_sms && $cita->paciente->celular != null){
            $this->enviar_sms($cita,$URL_COBRO);
        } else if ($enviar_correo && $cita->paciente->email != null ){
            $this->enviar_email($cita,$URL_COBRO);
        }
    }
    
    private function enviar_email($cita,$URL_COBRO){
        $email = $cita->paciente->email;
        $articulo = SesionUtils::armarArticulo($cita->id_suscripcion, $eliminarTildes=false, $isIndex=true);
        Yii::$app->mailer->compose('cobro-cita',['model' => $cita, 'articulo'=>$articulo, 'URL_COBRO'=>$URL_COBRO])
            ->setFrom(['citas@oliviahealth.com' => 'Olivia Health'])
            ->setTo($email)
            ->setSubject("Emisión de cobro")
            ->send();
        
    }
    
    private function enviar_sms($cita,$URL_COBRO){
        $username = \yii::$app->params["clickSendU"];
        $password = \yii::$app->params["clickSendP"];
        $csConfig = Configuration::getDefaultConfiguration()
                ->setUsername($username)
                ->setPassword($password);
        $csApi = new SMSApi(new Client(['verify' => false]),$csConfig);
        $smsCollector = new SmsMessageCollection();
        $paciente = $cita->paciente;
        $articulo = SesionUtils::armarArticulo($cita->id_suscripcion, $eliminarTildes=false, $isIndex=true);
        $mensaje = $articulo." te ha enviado un enlace de cobro. Haz click y realiza tu pago. ".$URL_COBRO;
        $sms = new SmsMessage();
        $sms->setBody($mensaje);
        $sms->setTo($paciente->celular);
        $sms->setSource("sdk");
        $smsCollector->setMessages([$sms]);
        $result = $csApi->smsSendPost($smsCollector);
        $this->guardarBitacora($cita->id, $cita->estado, false, true, BitacoraCitaMensaje::ES_COBRO);
    }
    
    
    private function guardarBitacora($idCita,$estado,$esMail,$esSMS, $tipo){
        $bitacoraCita = new BitacoraCitaMensaje();
        $bitacoraCita->fecha = \date("Y-m-d H:i:s");
        $bitacoraCita->id_cita = $idCita;
        $bitacoraCita->estado_cita = $estado;
        $bitacoraCita->es_correo = $esMail;
        $bitacoraCita->es_sms = $esSMS;
        $bitacoraCita->tipo = $tipo;
        $bitacoraCita->save();
    }
    
    /**
     * @param float $costo costo de cita
     * @param int $metodo_iva tipo de cargo de iva
     * @param int $opt_pasarela quien asume el cargo de la pasarela
     * @return array con los valores calculados
     */
    private function CALCULAR_TOTAL($costo,$metodo_iva,$opt_pasarela){
        $monto = 0;
        $iva = 0;
        $cargo_pasarela = 0;
        $total = 0;
        if ($metodo_iva == CobroPagadito::IVA_INCLUIDO && $opt_pasarela == CobroPagadito::ASUMIR_CARGO_PASARELA){
            $iva = $this->iva_incluido($costo);
            $cargo_pasarela = $this->calcular_cargo_pasarela($costo);
            $total = $costo;
            return $this->formato_respuesta($iva, $cargo_pasarela, $total);
        } else if ($metodo_iva == CobroPagadito::IVA_INCLUIDO && $opt_pasarela == CobroPagadito::CARGO_PASARELA_PACIENTE){
            $iva = $this->iva_incluido($costo);
            $costo = $this->iva_incluido_2($costo);
            $monto = $this->monto_cobrar_2($costo); 
            $cargo_pasarela = $this->calcular_cargo_pasarela($monto);
            $total = $monto; 
            return $this->formato_respuesta($iva, $cargo_pasarela, $total);
        } else if ($metodo_iva == CobroPagadito::AGREGAR_IVA && $opt_pasarela == CobroPagadito::ASUMIR_CARGO_PASARELA){
            $monto = $this->monto_cobrar_3($costo);
            $iva = $this->agregar_iva($costo);
            $cargo_pasarela = $this->calcular_cargo_pasarela($monto);
            $total = $monto;
            return $this->formato_respuesta($iva, $cargo_pasarela, $total);
        } else if ($metodo_iva == CobroPagadito::AGREGAR_IVA && $opt_pasarela == CobroPagadito::CARGO_PASARELA_PACIENTE){
            $monto = $this->monto_cobrar_2($costo);
            $iva = $this->iva_incluido($monto);
            $cargo_pasarela = $this->calcular_cargo_pasarela($monto);
            $total = $monto;
            return $this->formato_respuesta($iva, $cargo_pasarela, $total);
        } else if ($metodo_iva == CobroPagadito::SIN_IVA && $opt_pasarela == CobroPagadito::ASUMIR_CARGO_PASARELA){
            $cargo_pasarela = $this->calcular_cargo_pasarela($costo);
            $total = $costo;
            return $this->formato_respuesta(0, $cargo_pasarela, $total);
        } else if ($metodo_iva == CobroPagadito::SIN_IVA && $opt_pasarela == CobroPagadito::CARGO_PASARELA_PACIENTE){
            $monto = $this->monto_cobrar($costo);
            $cargo_pasarela = $this->calcular_cargo_pasarela($monto);
            $total = $monto;
            return $this->formato_respuesta(0, $cargo_pasarela, $total);
        }
    }
    
    /**
     * Unificar los valores de respuesta de los cargos
     * @param float $iva
     * @param int $cargo_pasarela
     * @param float $total
     * @return array
     */
    private function formato_respuesta($iva, $cargo_pasarela, $total){
        return [
            'iva' => $iva, 
            'cargo_pasarela' => $cargo_pasarela, 
            'total_cobrado' => round($total, 2)
        ];
    }
    
    /**
     * IVA = montoCobrar - montoCobrar/1.13 resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function iva_incluido($costo){
        return round($costo - ($costo / 1.13), 2);
    }
    
    /**
     * IVA =  montoCobrar/1.13 resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function iva_incluido_2($costo){
        return round(($costo / 1.13), 2);
    }
    
    /**
     * montoCobrar = montoServicios * 1.13 resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function agregar_iva($costo){
        return round(($costo * 0.13), 2);
    }
    
    /**
     * CargoPlataforma = ((montoCobrar*0.05)+0.25)*1.13 resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function calcular_cargo_pasarela($costo){
        return round(((($costo * 0.05) + 0.25)*1.13), 2);
    }

    /**
     * MontoCobrar = (MontoServicios+(0.25*1.13))/(1-(0.05*1.13)) resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function  monto_cobrar($costo){
        return round(($costo+(0.25*1.13)) / (1-(0.05*1.13)), 2 );
    }
    
    /**
     * MontoCobrar = ((MontoServicios*1.13)+(0.25*〖1.13〗^2 ))/(1-(0.05*〖1.13〗^2)) resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function monto_cobrar_2($costo){
        $pow = pow(1.13, 2);
        $num = ($costo*1.13) + (0.25*$pow);
        $den = 1-(0.05*$pow);
        return round($num / $den, 2);
    }

    /**
     * MontoCobrar = (MontoServicios*1.13) resultado aproximado
     * @param  float $costo costo de cita final
     * @return float
     */
    private function monto_cobrar_3($costo){
        return round($costo*1.13, 2);
    }
    

    private static function crypto_rand_secure($min, $max)
    {
        $range = $max - $min;
        if ($range < 1) return $min; // not so random...
        $log = ceil(log($range, 2));
        $bytes = (int) ($log / 8) + 1; // length in bytes
        $bits = (int) $log + 1; // length in bits
        $filter = (int) (1 << $bits) - 1; // set all lower bits to 1
        do {
            $rnd = hexdec(bin2hex(openssl_random_pseudo_bytes($bytes)));
            $rnd = $rnd & $filter; // discard irrelevant bits
        } while ($rnd > $range);
        return $min + $rnd;
    }
    
    /**
     * Obtener el número único de facturan 
     * @param  int $longitud longitud del número a generar 
     * @return string
     */
    private function NUM_FACTURA($longitud)
    {
        $N_FACT = "";
        $max = strlen(self::Code_N_Fact);
        for ($i=0; $i < $longitud; $i++) {
            $N_FACT .= self::Code_N_Fact[self::crypto_rand_secure(0, $max-1)];
        }

        return $N_FACT;
    }
    
    /**
     * Sí el precio de la cita cambia se recalcula el costo de los procedimientos
     * @param int $idCita
     */
    private function costoProcedimiento($cita){
        $procedimiento_cita = ProcedimientoCita::find()->where(['id_cita' => $cita->id])->all();
        $procedimientos = $this->procedimientoSimpleArray($procedimiento_cita);
        $costoProc = $this->costoSimpleArray($procedimiento_cita);
        $this->actualizarCostoPocedimientoCita(false,$cita->id,$procedimientos,$costoProc, $cita->costo);
    }
    
    private function procedimientoSimpleArray($procedimientos){
        $res = [];
        foreach($procedimientos as $proc){
            $res[] = $proc->procedimiento->id;
        }
        return $res;
    }
    
    private function costoSimpleArray($procedimientos){
        $res = [];
        foreach($procedimientos as $proc){
            $res[] = $proc->procedimiento->costo;
        }
        return $res;
    }
    
    private function actualizarCostoPocedimientoCita($isNewRecord,$idCita,$procedimientos,$costoProc, $costoCita){
        if(!$isNewRecord){
            ProcedimientoCita::deleteAll(['id_cita' => $idCita]);
        }
        $totalReal = $this->calcularTotalreal($costoProc);
        for($i=0; $i < count($procedimientos); $i++ ){
            $procedimientoCita = new ProcedimientoCita();
            $procedimientoCita->id_cita = $idCita;
            $procedimientoCita->id_procedimiento = $procedimientos[$i];
            $procedimientoCita->orden = $i;
            $procedimientoCita->costo_editado = $this->calcularDiferencia($totalReal,$costoCita,$costoProc[$i]);
            $procedimientoCita->save();
        }
    }
    /**
     * Sumar los costos de los procedimientos y calcular el total de la cita
     * @param array $costoProcedimiento
     * @return float 
     */
    private function calcularTotalreal($costoProcedimiento){
        $r = 0;
        foreach ($costoProcedimiento as $costo){
            $r += $costo;
        }
        return $r;
    }
    
    private function calcularDiferencia($totalReal,$costoCita,$costoP){
        return $totalReal == 0 ? 0 : ($costoP * $costoCita)/$totalReal;
    }
    
    
    /**
     * Finds the CobroPagadito model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Paciente the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = CobroPagadito::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }
    
    
    
}
